package com.framsticks.params;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;

import com.framsticks.util.FramsticksException;

import static com.framsticks.util.lang.Containers.*;

/**
 * The Class ReflectionAccess. Stores data in provided object using reflection.
 *
 * @author Mateusz Jarus <name.surname@gmail.com> (please replace name and
 *         surname with my personal data)
 *
 * @author Piotr Sniegowski
 */
public class ReflectionAccess extends SimpleAbstractAccess {
	private final static Logger log = Logger.getLogger(ReflectionAccess.class
			.getName());

	protected final Class<?> reflectedClass;
	private Object object;


	protected interface ReflectedSetter {
		public <T> void set(Object object, T value) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException;
	}

	protected interface ReflectedGetter {
		public abstract <T> T get(Object object, Class<T> type) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException;
	}

	protected static class ReflectedValueParam {
		public ReflectedSetter setter;
		public ReflectedGetter getter;
	}

	protected final Map<ValueParam, ReflectedValueParam> reflectedValueParams = new HashMap<>();

	public ReflectionAccess(Class<?> reflectedClass) throws ConstructionException {
		this(reflectedClass, FramsClassBuilder.buildForClass(reflectedClass));
	}

	public static boolean typeMatch(Class<?> a, Class<?> b) {
		assert !b.isPrimitive();
		if (!a.isPrimitive()) {
			return a.equals(b);
		}

		if (a.equals(int.class)) {
			return b.equals(Integer.class);
		}
		if (a.equals(double.class)) {
			return b.equals(Double.class);
		}
		if (a.equals(boolean.class)) {
			return b.equals(Boolean.class);
		}
		assert false;
		return false;
	}

	public ReflectionAccess(Class<?> reflectedClass, FramsClass framsClass) throws ConstructionException {
		this.reflectedClass = reflectedClass;
		setFramsClass(framsClass);

		Map<String, ParamCandidate> candidates = ParamCandidate.getAllCandidates(reflectedClass);

		try {
			for (final ValueParam vp : filterInstanceof(framsClass.getParamEntries(), ValueParam.class)) {
				if (!candidates.containsKey(vp.getId())) {
					throw new ConstructionException().msg("missing candidate for param").arg("param", vp);
				}
				ParamCandidate pc = candidates.get(vp.getId());
				if (pc.isReadOnly() && !vp.hasFlag(Flags.READONLY)) {
					throw new ConstructionException().msg("readonly state conflict").arg("param", vp);
				}
				if (!typeMatch(pc.getRawType(), vp.getStorageType())) {
					throw new ConstructionException().msg("types mismatch for param").arg("param", vp).arg("candidate", pc.getType()).arg("storage", vp.getStorageType());
				}

				ReflectedValueParam rvp = new ReflectedValueParam();
				reflectedValueParams.put(vp, rvp);
				final boolean primitive = pc.isPrimitive();
				if (pc.getField() != null) {
					final Field f = pc.getField();
					rvp.getter = new ReflectedGetter() {
						@Override
						public <T> T get(Object object, Class<T> type) throws IllegalArgumentException, IllegalAccessException {
							return type.cast(f.get(object));
						}
					};
					if (!pc.isFinal()) {
						rvp.setter = new ReflectedSetter() {
							@Override
							public <T> void set(Object object, T value) throws IllegalArgumentException, IllegalAccessException {
								if (value == null && primitive) {
									throw new FramsticksException().msg("setting null to primitive value");
								}
								f.set(object, value);
							}
						};
					}
				} else {
					final Method g = pc.getGetter();

					rvp.getter = new ReflectedGetter() {
						@Override
						public <T> T get(Object object, Class<T> type) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
							return type.cast(g.invoke(object));
						}
					};

					if (!pc.isFinal()) {
						final Method s = pc.getSetter();
						rvp.setter = new ReflectedSetter() {
							@Override
							public <T> void set(Object object, T value) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
								if (value == null && primitive) {
									throw new FramsticksException().msg("setting null to primitive value");
								}
								s.invoke(object, value);
							}
						};
					}
				}
			}
		} catch (ConstructionException e) {
			throw e.arg("java class", reflectedClass).arg("framsClass", framsClass);
		}
	}

	// private static String accessorName(boolean get, String id) {
	//	return (get ? "get" : "set") + id.substring(0, 1).toUpperCase() + id.substring(1);
	// }

	@Override
	public <T> T get(ValueParam param, Class<T> type) {
		try {
			try {
				if (object == null) {
					throw new FramsticksException().msg("no object set");
				}

				return reflectedValueParams.get(param).getter.get(object, type);
			} catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
				throw new FramsticksException().msg("failed to get").cause(e);
			}
		} catch (FramsticksException e) {
			throw e.arg("param", param).arg("type", type).arg("access", this);
		}
	}

	@Override
	protected <T> void internalSet(ValueParam param, T value) {
		setValue(param, value);
	}

	private <T> void setValue(ValueParam param, T value) {
		try {
			try {
				if (object == null) {
					throw new FramsticksException().msg("no object set");
				}
				ReflectedSetter s = reflectedValueParams.get(param).setter;
				if (s == null) {
					throw new FramsticksException().msg("trying to set final");
				}
				s.set(object, value);
			} catch (IllegalArgumentException | IllegalAccessException | InvocationTargetException e) {
				throw new FramsticksException().msg("failed to set").cause(e);
			}
		} catch (FramsticksException e) {
			throw e.arg("param", param).arg("value", value).arg("access", this);
		}
	}

	void resetErrors() {
		//TODO this replaces returnedObject.resetErrors();
	}

	@Override
	public void clearValues() {
		if (object == null) {
			return;
		}

		resetErrors();

		try {
			for (ValueParam p : filterInstanceof(framsClass.getParamEntries(), ValueParam.class)) {
				setValue(p, p.getDef(Object.class));
			}
		} catch (IllegalArgumentException ex) {
			ex.printStackTrace();
		}
	}

	/**
	 * Sets the new object to operate on.
	 *
	 * @param object
	 *            new object to operate on
	 */
	@Override
	public ReflectionAccess select(Object object) {
		assert object == null || reflectedClass.isInstance(object);
		this.object = object;
		return this;
	}

	@Override
	public Object getSelected() {
		return object;
	}

	// TODO: find a better place for it
	public static String objectToString(Object object) {
		StringBuilder b = new StringBuilder();
		for (Field f : object.getClass().getFields()) {
			b.append(f.getName());
			b.append(":");
			try {
				Object value = f.get(object);
				b.append((value != null) ? value.toString() : "<null>");
			} catch (IllegalAccessException e) {
				e.printStackTrace();
			}
			b.append("\n");
		}
		return b.toString();
	}


	@Override
	public ReflectionAccess cloneAccess() throws ConstructionException {
		return new ReflectionAccess(reflectedClass, framsClass);
	}

	@Override
	public Object createAccessee() {
		try {
			return reflectedClass.newInstance();
		} catch (InstantiationException | IllegalAccessException e) {
			e.printStackTrace();
		}
		log.fatal("failed to create reflected object of class " + reflectedClass.getCanonicalName() + " for frams type " + framsClass.getId());
		return null;
	}

	@Override
	public String toString() {
		StringBuilder b = new StringBuilder();
		b.append(framsClass);
		if (object != null) {
			b.append("(").append(object).append(")");
		}
		return b.toString();
	}

}

