package com.framsticks.params;

import java.io.InputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Member;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;

import com.framsticks.params.annotations.FramsClassAnnotation;
import com.framsticks.params.annotations.ParamAnnotation;
import com.framsticks.parsers.FileSource;
import com.framsticks.parsers.Loaders;
import com.framsticks.util.lang.Strings;

public class FramsClassBuilder {
	private static final Logger log =
		Logger.getLogger(FramsClassBuilder.class);


	public static String getName(FramsClassAnnotation fca, Class<?> javaClass) {
		return fca.name().equals("") ? javaClass.getSimpleName() : fca.name();
	}

	public static String getId(FramsClassAnnotation fca, Class<?> javaClass) {
		return fca.id().equals("") ? javaClass.getSimpleName() : fca.id();
	}

	public static String getParamTypeForNativeType(Type type) {
		if (type instanceof ParameterizedType) {
			ParameterizedType p = (ParameterizedType) type;
			Type rawType = p.getRawType();
			Type containedType = null;
			//TODO make implementation here
			boolean map = false;
			if (rawType.equals(Map.class)) {
				containedType = p.getActualTypeArguments()[1];
				map = true;
			} else if (rawType.equals(List.class)) {
				containedType = p.getActualTypeArguments()[0];
			}
			if (!(containedType instanceof Class)) {
				return null;
			}
			Class<?> containedClass = (Class<?>) containedType;
			StringBuilder b = new StringBuilder();
			b.append("l ");
			FramsClassAnnotation fca = containedClass.getAnnotation(FramsClassAnnotation.class);
			if (fca == null) {
				log.error("the class is not annotated: " + containedClass);
				return null;
			}
			b.append(getName(fca, containedClass));
			if (map) {
				b.append(" name");
			}

			return b.toString();
		}

		if (type.equals(Integer.class) || type.equals(int.class)) {
			return "d";
		}
		if (type.equals(String.class)) {
			return "s";
		}
		if (type.equals(Double.class) || type.equals(double.class)) {
			return "f";
		}
		if (type.equals(Boolean.class) || type.equals(boolean.class)) {
			return "d 0 1";
		}
		if (type.equals(Object.class)) {
			return "x";
		}
		if (type instanceof Class) {
			return "o " + ((Class<?>) type).getCanonicalName();
		}
		return null;
	}

	public static final String GENERATE_HELP_PREFIX = "automatically generated from: ";

	public static FramsClass readFromStream(InputStream stream) {
		return Loaders.loadFramsClass(new FileSource(stream));
	}

	// public static Class<? extends Param> getParamType(@Nonnull Class<?> c) {
	//	if (c.equals(Integer.class)) {
	//		return DecimalParam.class;
	//	}
	//	if (c.equals(Double.class)) {
	//		return FloatParam.class;
	//	}
	//	if (c.equals(String.class)) {
	//		return StringParam.class;
	//	}
	//	if (c.equals(Object.class)) {
	//		return UniversalParam.class;
	//	}
	//	return null;
	// }

	public static String extractIdOf(Member member) {
		if (member instanceof Field) {
			return member.getName();
		}
		if (member instanceof Method) {
			Method m = (Method) member;
			String n = m.getName();
			int argsNum = m.getParameterTypes().length;
			if (argsNum == 0) {
				return n.startsWith("get") ? Strings.uncapitalize(n.substring(3)) : n;
			}
			if (argsNum == 1) {
				return n.startsWith("set") ? Strings.uncapitalize(n.substring(3)) : n;
			}
			log.error("invalid number of arguments");
			return null;
		}
		log.error("invalid kind of member");
		return null;
	}
	public static String getName(ParamAnnotation annotation, Member member) {
		return annotation.name().equals("") ? Strings.capitalize(extractIdOf(member)) : annotation.name();
	}

	public static String getId(ParamAnnotation annotation, Member member) {
		return annotation.id().equals("") ? extractIdOf(member) : annotation.id();
	}

	public static ParamBuilder fill(ParamBuilder builder, Member member, ParamAnnotation annotation) {
		return builder
			.id(getId(annotation, member))
			.name(getName(annotation, member));

	}

	public static FramsClass buildForClass(Class<?> javaClass) throws ConstructionException {

		FramsClassAnnotation fca = javaClass.getAnnotation(FramsClassAnnotation.class);
		if (fca == null) {
			log.error(javaClass.getName() + " is not annotated with FramsClassAnnotation");
			return null;
		}

		FramsClass framsClass = new FramsClass(getId(fca, javaClass), getName(fca, javaClass), "");

		Map<String, ParamCandidate> candidates = ParamCandidate.getAllCandidates(javaClass);

		for (ParamCandidate pc : candidates.values()) {
			framsClass.append(Param.build().id(pc.getId()).name(pc.getName()).type(getParamTypeForNativeType(pc.getType())).flags(pc.getFlags()));
		}

		return framsClass;
	}
}
