package com.framsticks.params;

import com.framsticks.params.annotations.FramsClassAnnotation;
import com.framsticks.params.annotations.ParamAnnotation;
// import com.framsticks.util.FramsticksException;

import java.util.*;

import javax.annotation.Nonnull;

/**
 * The class FramsClass represents the class / schema of connected parameters
 * (such as parameters within the class). It differs from C++ version by storing
 * information about the class that parameters belong to.
 *
 * Based loosely on c++ class Param located in cpp/gdk/param.*
 *
 * @author Jarek Szymczak <name.surname@gmail.com>, Mateusz Jarus (please
 *         replace name and surname with my personal data)
 *
 * @author Piotr Sniegowski
 */
@FramsClassAnnotation(id = "class", name = "class")
public final class FramsClass {

	/** The offset of the parameter (applied for newly added parameter). */
	protected int fieldsNumber;

	/** The groups. */
	protected List<Group> groups = new ArrayList<Group>();

	/**
	 * The param entry map <parameterId, param> (for fast accessing of parameters
	 * by their name)
	 */
	protected Map<String, Param> paramEntryMap = new LinkedHashMap<String, Param>();

	/** The param list (for accessing parameters by offset in O(1) time. */
	protected List<Param> paramList = new ArrayList<Param>();

	/** The param getId map (for fast lookup of offset based on name */
	protected Map<String, Integer> paramIdMap = new HashMap<String, Integer>();

	protected String id;

	protected String name;

	protected String description;

	public Collection<Param> getParamEntries() {
		return paramList;
	}

	public FramsClass() {

	}

	public FramsClass(String id, String name, String description) {
		this.setId(id);
		this.setName(name);
		this.setDescription(description);
	}

	/**
	 * Adds new param entry.
	 *
	 * @param param
	 *            the new param entry
	 */
	public FramsClass append(Param param) {
		// if (param.hasFlag(Flags.USERHIDDEN)) {
		// 	return this;
		// }
		paramEntryMap.put(param.getId(), param);
		//paramEntryMap.put(param.getInternalId(), param);
		paramList.add(param);
		try {
			Group group = groups.get(param.getGroup());
			if (group != null) {
				group.addProperty(param);
			}
		} catch (IndexOutOfBoundsException ignored) {

		}

		return this;
	}

	public FramsClass append(ParamBuilder builder) {
		return append(builder.finish());
	}

	/**
	 * Adds new group.
	 */
	public FramsClass appendGroup(Group group) {
		groups.add(group);
		return this;
	}

	@ParamAnnotation(id = "desc")
	public String getDescription() {
		return description;
	}

	public int getGroupCount() {
		return groups.size();
	}

	/**
	 * Gets the group member.
	 *
	 * @param gi
	 *            the offset of group
	 * @param pi
	 *            the offset of member within a group
	 * @return the pi-th member of group gi
	 */
	public Param getGroupMember(int gi, int pi) {
		if (gi < 0 || pi < 0 || gi >= groups.size()) {
			return null;
		}
		Group group = groups.get(gi);
		return (group != null ? group.getProperty(pi) : null);
	}

	/**
	 * Gets the group name.
	 *
	 * @param gi
	 *            the offset of group
	 * @return the group name
	 */
	public String getGroupName(int gi) {
		if (gi < 0 || gi >= groups.size())
			return null;
		return groups.get(gi).name;
	}

	@ParamAnnotation
	public String getId() {
		return id;
	}

	@ParamAnnotation
	public String getName() {
		return name;
	}

	public String getNiceName() {
		return name != null ? name : id;
	}

	public <T extends Param> T castedParam(@Nonnull final Param param, @Nonnull final Class<T> type, Object name) {
		if (param == null) {
			return null;
			// throw new FramsticksException().msg("param is missing").arg("name", name).arg("in", this);
		}
		if (!type.isInstance(param)) {
			return null;
			// throw new FramsticksException().msg("wrong type of param").arg("actual", param.getClass()).arg("requested", type).arg("in", this);
		}
		return type.cast(param);
	}

	/**
	 * Gets the param entry.
	 *
	 * @param i
	 *            the offset of parameter
	 * @return the param entry
	 */
	public <T extends Param> T getParamEntry(final int i, @Nonnull final Class<T> type) {
		return castedParam(getParam(i), type, i);
	}

	/**
	 * Gets the param entry.
	 *
	 * @param id
	 *            the getId of parameter
	 * @return the param entry
	 */
	public <T extends Param> T getParamEntry(@Nonnull final String id, @Nonnull final Class<T> type) {
		return castedParam(getParam(id), type, id);
	}

	public Param getParam(int i) {
		if (i < 0 || i >= paramList.size()) {
			return null;
		}
		return paramList.get(i);
	}

	public Param getParam(String id) {
		if (!paramEntryMap.containsKey(id)) {
			return null;
		}
		return paramEntryMap.get(id);
	}

	public int getParamCount() {
		return paramList.size();
	}

	@Override
	public String toString() {
		return id;
	}

	@ParamAnnotation
	public void setId(String id) {
		this.id = id;
	}

	@ParamAnnotation
	public void setName(String name) {
		this.name = name;
	}

	@ParamAnnotation(id = "desc")
	public void setDescription(String description) {
		this.description = description;
	}
}
