package com.framsticks.core;

import javax.annotation.OverridingMethodsMustInvokeSuper;

import com.framsticks.params.annotations.FramsClassAnnotation;
import com.framsticks.params.annotations.ParamAnnotation;
import com.framsticks.util.dispatching.Thread;
import com.framsticks.util.dispatching.Dispatcher;

import org.apache.commons.configuration.Configuration;
import org.apache.log4j.Logger;
import com.framsticks.util.dispatching.RunAt;

/**
 * @author Piotr Sniegowski
 */
@FramsClassAnnotation
public abstract class Entity implements Dispatcher<Entity> {

	private final static Logger log = Logger.getLogger(Entity.class);

	@ParamAnnotation
	protected String name = "entity";
	protected EntityOwner owner;
	protected Dispatcher<Entity> dispatcher;

	public Entity() {
	}

	public final String getName() {
		return name;
	}

	/**
	 * @param name the name to set
	 */
	public final void setName(String name) {
		this.name = name;
		if (dispatcher instanceof Thread) {
			((Thread<Entity>) dispatcher).setName(name);
		}
	}

	/**
	 * @return the owner
	 */
	public EntityOwner getOwner() {
		return owner;
	}

	/**
	 * @param owner the owner to set
	 */
	public void setOwner(EntityOwner owner) {
		this.owner = owner;
	}

	@Override
	public final boolean isActive() {
		return dispatcher == null || dispatcher.isActive();
	}

	@Override
	public final void invokeLater(RunAt<? extends Entity> runnable) {
		assert dispatcher != null;
		dispatcher.invokeLater(runnable);
	}

	public Dispatcher<Entity> createDefaultDispatcher() {
		return new Thread<Entity>(name).start();
	}

	@OverridingMethodsMustInvokeSuper
	protected void run() {
		assert isActive();
		log.info("running: " + this);
	}


	public void configure(Configuration config) {
	}

	public Dispatcher<Entity> getDispatcher() {
		return dispatcher;
	}

	/**
	 * @param dispatcher the dispatcher to set
	 */
	public void setDispatcher(Dispatcher<Entity> dispatcher) {
		this.dispatcher = dispatcher;
	}

	public final void start() {
		if (dispatcher == null) {
			log.debug("no dispatcher set for " + this + ", creating default one");
			setDispatcher(createDefaultDispatcher());
		}
		invokeLater(new RunAt<Entity>() {
			@Override
			public void run() {
				Entity.this.run();
			}
		});
	}

	public final void done() {
		log.info("stopping entity");
		if (owner != null) {
			owner.onDone();
		}
	}

}
