package com.framsticks.parsers;

import org.testng.annotations.*;
import com.framsticks.model.*;
import com.framsticks.model.Package;
import com.framsticks.params.*;
import com.framsticks.params.types.FloatParam;
import com.framsticks.test.TestConfiguration;
import com.framsticks.util.math.Point3d;

import java.io.IOException;
import java.text.ParseException;
import java.util.List;
import static org.fest.assertions.Assertions.*;
import static org.fest.assertions.Delta.*;

/**
 * Author: Piotr Śniegowski
 */
public class F0ParserTest extends TestConfiguration {

	private Schema schema;
	private List<AccessInterface> accesses;
	private List<Object> objects;
	private Model model;

	@BeforeClass
	public void setUp() throws Exception {
		schema = new Schema(Schema.getDefaultDefinitionAsStream());
		Package.register(schema.getRegistry());
	}

	@Test
	public void primitiveParam() {
		FramsClass joint = schema.getRegistry().getInfoFromCache("j");
		PrimitiveParam dx = joint.getParamEntry("dx", PrimitiveParam.class);
		assertThat(dx).isInstanceOf(FloatParam.class);
		assertThat(schema.getNeuroClasses().size()).isEqualTo(21);
		assertThat(dx.getName()).isEqualTo("delta.x");
		assertThat(dx.getMin(Double.class)).isEqualTo(-2.0, delta(0.0));
	}

	@Test
	public void readF0() throws IOException, ParseException {
		accesses = new F0Parser(schema, F0ParserTest.class.getResourceAsStream("/parsers/f0_example.txt")).parse();

		assertThat(accesses.size()).isEqualTo(12);
		assertThat(accesses.get(0).getSelected()).isInstanceOf(Model.class);
		assertThat(accesses.get(5).get("i", String.class)).isEqualTo("1,2,3,\"dsadsa,,,,");
		assertThat(accesses.get(7).get("d", String.class)).isEqualTo("|:p=0.25,r=1");
		assertThat(accesses.get(10).get("d", String.class)).isEqualTo("@:p=0.25");
	}

	@Test(dependsOnMethods = {"readF0"})
	public void stripAccessInterface() {
		objects = Util.stripAccessInterface(accesses);

		assertThat(objects.get(1)).isInstanceOf(Part.class);
		assertThat(objects.get(4)).isInstanceOf(Joint.class);
		assertThat(objects.get(6)).isInstanceOf(NeuroDef.class);
	}

	@Test(dependsOnMethods = {"stripAccessInterface"})
	public void buildModel() {
		model = Model.build(objects);

		assertThat(model.getParts().size()).isEqualTo(3);
		assertThat(model.getNeuroDefs().size()).isEqualTo(6);
		assertThat(model.getJoints().size()).isEqualTo(2);

		assertThat(model.getJoints().get(0).part1).isEqualTo(0);
		assertThat(model.getJoints().get(0).part2).isEqualTo(1);
		assertThat(model.getNeuroDefs().get(0).part).isEqualTo(1);
		assertThat(model.getNeuroDefs().get(0).joint).isEqualTo(-1);
		assertThat(model.getNeuroDefs().get(1).details).isEqualTo("|:p=0.25,r=1");
		assertThat(model.getNeuroDefs().get(3).details).isEqualTo("N");
		assertThat(model.getNeuroDefs().get(4).part).isEqualTo(-1);

		assertThat(model.getParts().get(1).getPosition().x).isEqualTo(2.0, delta(0.0));
		assertThat(model.getParts().get(2).getPosition().sub(new Point3d(2.27236, -0.0792596, -0.958924)).length()).isLessThan(0.0001);
		assertThat(model.getParts().get(2).getOrientation().y.sub(new Point3d(0.870277, -0.404792, 0.280644)).length()).isLessThan(0.0001);
	}

	@Test(dependsOnMethods = {"buildModel"})
	public void print() throws Exception {
		ListSink sink = new ListSink();
		new F0Writer(schema, model, sink).write();

		assertThat(sink.getOut()).containsExactly(
			"p:",
			"p:2.0,i=,Vstyle=",
			"p:2.272364001928095,-0.07925961087140347,-0.9589242746631385,i=bla",
			"j:0,1,dx=2.0",
			"j:1,2,rx=8.0,5.0,6.0,dx=1.0,i=\"1,2,3,\\\"dsadsa,,,,\"",
			"n:p=1",
			"n:j=0,d=\"|:p=0.25,r=1\"",
			"n:j=0,d=G",
			"n:p=1",
			"n:j=0,d=@:p=0.25",
			"n:p=1,d=Nu",
			"m:"
		);

	}
}
