package com.framsticks.parsers;

import com.framsticks.model.*;
import com.framsticks.model.Package;
import com.framsticks.params.*;
import com.framsticks.params.types.FloatParam;
import com.framsticks.util.Point3d;
import org.apache.log4j.PropertyConfigurator;
import org.junit.Test;

import java.util.List;

import static org.junit.Assert.*;

/**
 * Author: Piotr Śniegowski
 */
public class F0ParserTest {

	@Test
	public void testParser() throws Exception {
		PropertyConfigurator.configure(getClass().getResource("/log4j.properties"));
		//testing schema
		Schema schema = new Schema(Schema.getDefaultDefinitionAsStream());

		{
			FramsClass joint = schema.getRegistry().getInfoFromCache("j");
			Param dx = joint.getParamEntry("dx");
			assertEquals(FloatParam.class, dx.getClass());
			assertEquals(21, schema.getNeuroClasses().size());
			assertEquals("delta.x", dx.getName());
			assertEquals(-2.0, dx.getMin(Double.class), 0.0);
		}

		Package.register(schema.getRegistry());
		//testing parser
		List<AccessInterface> accesses = new F0Parser(schema, F0ParserTest.class.getResourceAsStream("/parsers/f0_example.txt")).parse();

		{
			assertEquals(12, accesses.size());
			assertTrue(accesses.get(0).getSelected() instanceof F0Genotype);
			assertEquals("1,2,3,\"dsadsa,,,,", accesses.get(5).get("i", String.class));
			assertEquals("|:p=0.25,r=1", accesses.get(7).get("d", String.class));
			assertEquals("@:p=0.25", accesses.get(10).get("d", String.class));
		}

		List<Object> objects = Util.stripAccessInterface(accesses);
		{
			assertEquals(Part.class, objects.get(1).getClass());
			assertEquals(Joint.class, objects.get(4).getClass());
			assertEquals(NeuroDef.class, objects.get(6).getClass());
		}

		F0Genotype f0Genotype = Genotype.build(objects);
		{
			assertEquals(3, f0Genotype.getParts().size());
			assertEquals(6, f0Genotype.getNeuroDefs().size());
			assertEquals(2, f0Genotype.getJoints().size());

			assertEquals(new Integer(0), f0Genotype.getJoints().get(0).part1);
			assertEquals(new Integer(1), f0Genotype.getJoints().get(0).part2);
			assertEquals(new Integer(1), f0Genotype.getNeuroDefs().get(0).part);
			assertEquals(new Integer(-1), f0Genotype.getNeuroDefs().get(0).joint);
			assertEquals("|:p=0.25,r=1", f0Genotype.getNeuroDefs().get(1).details);
			assertEquals("N", f0Genotype.getNeuroDefs().get(3).details);
			assertEquals(new Integer(-1), f0Genotype.getNeuroDefs().get(4).part);
		}

		F0Model f0model = F0Model.build(f0Genotype);
		{
			assertEquals(2.0, f0model.getMechParts().get(1).getPosition().x, 0.0);
			assertTrue(f0model.getMechParts().get(2).getPosition().sub(new Point3d(2.27236, -0.0792596, -0.958924)).length() < 0.0001);
			assertTrue(f0model.getMechParts().get(2).getOrientation().y.sub(new Point3d(0.870277, -0.404792, 0.280644)).length() < 0.0001);
		}

	}
}
