package com.framsticks.params;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.util.List;

import com.framsticks.params.types.ListParam;
import org.apache.log4j.Logger;


/**
 * The Class ReflectionAccess. Stores data in provided object using reflection.
 * 
 * @author Mateusz Jarus <name.surname@gmail.com> (please replace name and
 *         surname with my personal data)
 *
 * @author Piotr Sniegowski
 */
public class ReflectionAccess extends SimpleAbstractAccess {
	private final static Logger LOGGER = Logger.getLogger(ReflectionAccess.class
			.getName());


    protected final Class reflectedClass;
	private Object object;

	public ReflectionAccess(Class reflectedClass, FramsClass framsClass) {
        this.reflectedClass = reflectedClass;
		setFramsClass(framsClass);
	}

	private static String accessorName(boolean get, String id) {
		return (get ? "get"  : "set") + id.substring(0, 1).toUpperCase() + id.substring(1);
	}

	@Override
	public <T> T get(Param param, Class<T> type) {
		if (object == null) {
			return null;
		}
		try {
			//TODO: use internal id, if present
			String id = param.getId();
			try {
				return type.cast(reflectedClass.getField(id).get(object));
			} catch (NoSuchFieldException ignored) {
			}
			try {
				return type.cast(reflectedClass.getMethod(accessorName(true, id)).invoke(object));
			} catch (NoSuchMethodException ex) {
				//ex.printStackTrace();
			} catch (InvocationTargetException e) {
				//e.printStackTrace();
            }

		} catch (IllegalAccessException ex) {
			LOGGER.warn("illegal access error occurred while trying to access returnedObject");
			ex.printStackTrace();
        } catch (ClassCastException ignored) {

        }
		return null;
	}

	@Override
	protected <T> void internalSet(Param param, T value) {
		setValue(param, value);
	}

	private <T> void setValue(Param param, T value) {
		if (object == null) {
			return;
        }
		try {
			String id = param.getId();
			try {
				Field f = reflectedClass.getField(id);
				Class t = f.getType();
				if (Modifier.isFinal(f.getModifiers())) {
					return;
				}
				if (value != null || (!t.isPrimitive())) {
					f.set(object, value);
				}
				return;
			} catch (NoSuchFieldException ignored) {
			}
			try {
				reflectedClass.getMethod(accessorName(false, id), new Class[]{param.getStorageType()}).invoke(object, value);
			} catch (InvocationTargetException ignored) {
			} catch (NoSuchMethodException ignored) {
			}
		} catch (Exception ex) {
			ex.printStackTrace();
		}
	}

	void resetErrors() {
		//TODO this replaces returnedObject.resetErrors();
	}

	@Override
	public void clearValues() {
		if (object == null) {
			return;
		}

		resetErrors();

		try {
			for (Param p : framsClass.getParamEntries()) {
				setValue(p, p.getDef(Object.class));
			}
		} catch (IllegalArgumentException ex) {
			ex.printStackTrace();
		}
	}

	/**
	 * Sets the new object to operate on.
	 * 
	 * @param object
	 *            new object to operate on
	 */
	@Override
	public void select(Object object) {
		assert object == null || reflectedClass.isInstance(object);
		this.object = object;
	}

	@Override
	public Object getSelected() {
		return object;
	}

	// TODO: find a better place for it
	public static String objectToString(Object object) {
		StringBuilder b = new StringBuilder();
		for (Field f : object.getClass().getFields()) {
			b.append(f.getName());
			b.append(":");
			try {
				Object value = f.get(object);
				b.append((value != null) ? value.toString() : "<null>");
			} catch (IllegalAccessException e) {
				e.printStackTrace();
			}
			b.append("\n");
		}
		return b.toString();
	}


	@Override
	public ReflectionAccess cloneAccess() {
		return new ReflectionAccess(reflectedClass, framsClass);
	}

    @Override
    public Object createAccessee() {
        try {
            return reflectedClass.newInstance();
        } catch (InstantiationException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        }
        LOGGER.fatal("failed to create reflected object of class " + reflectedClass.getCanonicalName() + " for frams type " + framsClass.getId());
        return null;
    }
}
