package com.framsticks.params;

import com.framsticks.params.types.DecimalParam;
import com.framsticks.params.types.StringParam;

import java.util.HashMap;
import java.util.Map;

/**
 * Based on c++ struct ParamEntry located in cpp/gdk/param.h
 * Here  it is a root of Param hierarchy.
 *
 * @author Jarek Szymczak <name.surname@gmail.com>, Mateusz Jarus
 * (please replace name and surname with my personal data)
 *
 * @author Piotr Śniegowski
 */
public abstract class Param {

	/** The parameter id. */
	protected String id;

	/**
	 * The parameter internal id. It's set by a user to use user's own getId in
	 * code
	 */
	protected String internalId;

	/** The parameter name. */
	protected String name;

	/** The help (description) concerning parameter. */
	protected String help;

	/** The number of group, that parameter belongs to. */
	protected Integer group;

	/** The getFlags stored as a bit sum. */
	protected Integer flags;

	/** The variable determining whether the parameter is an extra parameter. */
	protected Integer extra;

	/** The minimum allowed value of parameter. */
	protected Object min;

	/** The maximum allowed value of parameter. */
	protected Object max;

	/** The default value of parameter. */
	protected Object def;



	/**
	 * The defaults used for Integer, Double and String classes, if the default
	 * value of parameter is not explicitly given.
	 */
	@SuppressWarnings("rawtypes")
	private Map<Class, Object> defaults = new HashMap<Class, Object>();
	// static initialization of defaults
	{
		defaults.put(String.class, "");
		defaults.put(Integer.class, 0);
		defaults.put(Double.class, 0.0);
	}

	public Param() {

	}

	public String getId() {
		return id;
	}

	public String getInternalId() {
		return internalId;
	}

	public String getName() {
		return name;
	}

	public String getHelp() {
		return help;
	}

	public Integer getGroup() {
		return group;
	}

	public Integer getFlags() {
		return flags;
	}

    public abstract String getType();


	public String getEffectiveId() {
		return (internalId != null) ? internalId : id;
	}


    private <T> T tryCastAndReturn(Object value, Class<T> type) {
        if (value == null)
            return null;
        try {
            return type.cast(value);
        } catch (ClassCastException e) {
            throw new ClassCastException("property \"" + name
                    + "\" type is \"" + value.getClass().getName()
                    + "\", not \"" + type.getName() + "\"");
        }
    }
	/**
	 * Gets the minimum value of parameter.
	 *
	 * @param <T> the generic getType which must be correctly specified by user
	 * @param type the getType of variable, can be checked by
	 * @return the minimum allowed value of parameter
	 * @throws ClassCastException the class cast exception, thrown when given getType is incorrect
	 */
	public <T> T getMin(Class<T> type) {
		return tryCastAndReturn(min, type);
	}

	/**
	 * Gets the maximum value of parameter.
	 *
	 * @param <T> the generic getType which must be correctly specified by user
	 * @param type the getType of variable, can be checked by {@link #getStorageType()}
	 * @return the maximum allowed value of parameter
	 * @throws ClassCastException the class cast exception, thrown when given getType is incorrect
	 */
	public <T> T getMax(Class<T> type) {
		return tryCastAndReturn(max, type);
	}

	/**
	 * Gets the default value of parameter.
	 *
	 * @param <T> the generic getType which must be correctly specified by user
	 * @param type the getType of variable, can be checked by {@link #getStorageType()}
	 * @return the default value of parameter
	 * @throws ClassCastException the class cast exception, thrown when given getType is incorrect
	 */
	public <T> T getDef(Class<T> type) throws ClassCastException {
		return tryCastAndReturn(def, type);
	}

	public Integer getExtra() {
		return extra;
	}

	public void setInternalId(String internalId) {
		this.internalId = internalId;
	}

	@Override
	public String toString() {
		return getId() + ":" + this.getClass().getSimpleName();
	}

	public boolean isNumeric() {
		return false;
	}

	public boolean isEmptyAvailable() {
		return false;
	}

	public abstract Class getStorageType();

	public boolean hasFlag(int flag) {
		return flags != null && (flags & flag) != 0;
	}

	public Object reassign(Object newValue, Object oldValue) throws CastFailure {
        if (newValue == null) {
            if (isEmptyAvailable()) {
                return null;
            }
            throw new CastFailure();
        }
        try {
            return getStorageType().cast(newValue);
        } catch (ClassCastException ignored) {
            throw new CastFailure();
        }
    }

	public boolean isUserHidden() {
		return (flags & Flags.USERHIDDEN) != 0;
	}

    public static FramsClass getFramsClass() {
        return new FramsClass("prop", "prop", null)
                .append(new ParamBuilder().setId("name").setName("Name").setType(StringParam.class).build())
                .append(new ParamBuilder().setId("id").setName("Id").setType(StringParam.class).build())
                .append(new ParamBuilder().setId("type").setName("Type").setType(StringParam.class).build())
                .append(new ParamBuilder().setId("help").setName("Help").setType(StringParam.class).build())
                .append(new ParamBuilder().setId("flags").setName("Flags").setType(DecimalParam.class).build());
    }

}
