package com.framsticks.gui;

import com.framsticks.communication.Subscription;
import com.framsticks.communication.util.LoggingStateCallback;
import com.framsticks.core.ListChange;
import com.framsticks.core.Path;
import com.framsticks.gui.components.ValueControl;
import com.framsticks.gui.view.TreeCellRenderer;
import com.framsticks.params.AccessInterface;
import com.framsticks.params.Param;
import com.framsticks.params.types.*;
import com.framsticks.remote.*;
import com.framsticks.util.Casting;
import com.framsticks.util.Future;
import com.framsticks.util.Logging;
import com.framsticks.util.StateFunctor;
import org.apache.log4j.Logger;

import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreePath;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * @author Piotr Sniegowski
 */
public class TreeNode extends DefaultMutableTreeNode implements NodeListener {

	private static final Logger LOGGER = Logger.getLogger(TreeNode.class.getName());

	Panel panel;

	final protected Frame frame;
    final protected EndpointAtFrame endpoint;

    final protected Map<EventParam, Subscription> userSubscriptions = new HashMap<EventParam, Subscription>();
	protected Map<ValueControl, Object> changedValues = null;
    protected String tooltip;
    protected String name;
    protected String iconName;
    protected Path path;


    public TreeNode(Frame frame, EndpointAtFrame endpoint, final Path path) {
        assert frame.isActive();
        this.frame = frame;
		LOGGER.debug("creating treenode for: " + path);
        this.path = path;
        this.endpoint = endpoint;

        name = path.getTop().getParam().getId();
        iconName = TreeCellRenderer.findIconName(name, path.getTextual());
        tooltip = "?";
        path.getInstance().invokeLater(new Runnable() {
            @Override
            public void run() {
                updateDescriptions(path);
            }
        });
	}

    public void fetch(final Path p) {
        assert p.getInstance().isActive();
        LOGGER.debug("fetching: " + p);
        p.getInstance().fetchValues(p, new StateFunctor() {
            @Override
            public void call(Exception e) {
                assert p.getInstance().isActive();
                if (Logging.log(LOGGER, "fetch", TreeNode.this, e)) {
                    frame.invokeLater(new Runnable() {
                        @Override
                        public void run() {
                            LOGGER.debug("removing node from tree: " + p);
                            TreeNode.this.removeFromParent();
                            frame.repaintTree();
                        }
                    });
                    return;
                }
                updateChildren(p);
                frame.invokeLater(new Runnable() {
                    @Override
                    public void run() {
                        //TODO maybe this should be called from some better place
                        useOrCreatePanel();
                    }
                });

            }
        });
    }

    protected void postUpdatePath(final Path newPath) {
        assert !frame.isActive();
        /** TODO those two actions could be merged into single closure */
        frame.invokeLater(new Runnable() {
            @Override
            public void run() {
                updatePath(newPath);
            }
        });
        updateDescriptions(newPath);
    }

    protected void updatePath(Path newPath) {
        assert frame.isActive();
        if (!path.isResolved()) {
            path = newPath;
			LOGGER.debug("updated treenode's path: " + path);
            return;
        }
        if (path.matches(newPath)) {
            return;
        }
        this.removeAllChildren();
    }

    /** Update children, by removing non existing and adding new ones.
     */
    protected void updateChildren(final Path p) {
        assert p.getInstance().isActive();
        LOGGER.debug("updating children of " + this);
        AccessInterface access = p.getInstance().bindAccess(p.getTop());
		if (access == null) {
			return;
		}
        final List<Path> childrenPaths = new LinkedList<Path>();
        /**Prepare path for each child.*/
        for (Param param : access.getParams()) {
            if (!(param instanceof CompositeParam)) {
                continue;
            }
            Path childPath = p.appendParam((CompositeParam)param);
            childrenPaths.add(childPath);
        }
        /**If some child were found, update in frame context.*/
        if (childrenPaths.size() > 0) {
            frame.invokeLater(new Runnable() {
                @Override
                public void run() {
                    TreePath treePath = frame.startChange();
                    updatePath(p);
                    Map<String, TreeNode> current = new HashMap<String, TreeNode>();
                    for (int c = 0; c < TreeNode.this.getChildCount(); ++c) {
                        TreeNode n = (TreeNode)TreeNode.this.getChildAt(c);
                        current.put(n.path.getLastElement(), n);
                    }
                    for (final Path childPath : childrenPaths) {
                        String e = childPath.getLastElement();
                        if (current.containsKey(e)) {
                            /*
                            final TreeNode n = current.get(e);
                            childPath.getInstance().invokeLater(new Runnable() {
                                @Override
                                public void run() {
                                    n.updateChildren(childPath);
                                }
                            });
                            */
                            current.remove(e);
                            continue;
                        }
                        TreeNode childNode = new TreeNode(frame, endpoint, childPath);
                        TreeNode.this.add(childNode);
                    }
                    for (TreeNode n : current.values()) {
                        TreeNode.this.remove(n);
                    }
                    frame.markNodeChanged(TreeNode.this, treePath);
                    frame.repaintTree();
                    LOGGER.debug("updated children of " + TreeNode.this);
                }
            });
        } else {
            LOGGER.debug("no children in " + TreeNode.this);
        }
    }

    public void select() {
        final Path p = path;

        p.getInstance().invokeLater(new Runnable() {
            @Override
            public void run() {
                final Path updated = (p.isResolved()) ? p : p.findResolution();
                if (updated.isResolved()) {
                    Logging.log(LOGGER, "update", updated, null);
                    fetch(updated);
                    postUpdatePath(updated);
                    return;
                }
                p.getInstance().resolve(updated, new Future<Path>() {
                    @Override
                    public void result(final Path result, Exception e) {
                        if (Logging.log(LOGGER, "resolve and select", TreeNode.this, e)) {
                            return;
                        }

                        fetch(result);
                        postUpdatePath(result);
                    }
                });
            }
        });

    }

    @Override
    public String toString() {
        return path.toString();
    }

	public final Panel getPanel() {
        assert frame.isActive();
		return panel;
	}

    protected void updateDescriptions(final Path p) {
        assert p.getInstance().isActive();
        if (!p.isResolved()) {
            return;
        }
        AccessInterface access = p.getInstance().bindAccess(p);
		if (access == null) {
			return;
		}
        StringBuilder t = new StringBuilder();
        /** html formatting is used here, since tooltips in swing do not support simple \n line breaks */
        t.append("<html>");
        t.append("frams: ").append(access.getId()).append("<br/>");
        t.append("java: ").append(p.getTopObject().getClass().getCanonicalName()).append("<br/>");
        t.append("access: ").append(access.getClass().getSimpleName());
        t.append("</html>");
        final String tooltip = t.toString();

        StringParam nameParam = Casting.tryCast(StringParam.class, access.getParam("name"));
        final String name = (nameParam != null ? access.get(nameParam, String.class) : path.getTop().getParam().getId());

        frame.invokeLater(new Runnable() {
            @Override
            public void run() {

                TreeNode.this.tooltip = tooltip;
                TreeNode.this.name = name;
            }
        });
    }

/*
	public void updateData() {
        assert browser.isActive();
		final Node node = getNode();
        browser.manager.invokeLater(new Runnable() {
            @Override
            public void run() {
                node.fetchValues(new StateFunctor() {
                    @Override
                    public void call(Exception e) {
                        if (e != null) {
                            return;
                        }
                        browser.invokeLater(new Runnable() {
                            @Override
                            public void run() {
                                assert browser.isActive();
                                if (panel.getCurrentTreeNode() == TreeNode.this) {
                                    panel.refreshComponents();
                                }

                                browser.tree.repaint();
                                panel.refreshComponents();
                            }
                        });
                    }
                });
            }
        });
	}
*/

    public void showPanel() {
        assert frame.isActive();
        assert panel != null;
        frame.showPanel(panel);
    }

    public void fillPanelWithValues() {
        assert frame.isActive();
        final Path p = path;
        assert p.isResolved();
        assert panel != null;
        panel.setCurrentTreeNode(this);
        p.getInstance().invokeLater(new Runnable() {
            @Override
            public void run() {
                AccessInterface access = p.getInstance().bindAccess(p);
                panel.refreshComponents(access);

                frame.invokeLater(new Runnable() {
                    @Override
                    public void run() {
                        showPanel();
                    }
                });
            }
        });

    }

	public void useOrCreatePanel() {
        assert frame.isActive();
		if (panel != null) {
            LOGGER.trace("panel is already attached: " + path);
            fillPanelWithValues();
			return;
		}
        if (!path.isResolved()) {
            LOGGER.trace("path is not resolved: " + path);
            return;
        }

        CompositeParam param = path.getTop().getParam();
        panel = endpoint.findPanel(param.computeAccessId());
        if (panel != null) {
            LOGGER.debug("found prepared panel for: " + path);
            fillPanelWithValues();
            return;
        }
        final Path p = path;
        LOGGER.debug("preparing panel: " + p);
        p.getInstance().invokeLater(new Runnable() {
            @Override
            public void run() {
                assert p.getInstance().isActive();
                final CompositeParam param = p.getTop().getParam();
                if (param instanceof ObjectParam) {
                    AccessInterface access = p.getInstance().prepareAccess(param);
                    final List<Param> params = new LinkedList<Param>();
                    for (Param p : access.getParams()) {
                        if (p instanceof CompositeParam) {
                            continue;
                        }
                        params.add(p);
                    }
                    frame.invokeLater(new Runnable() {
                        @Override
                        public void run() {
                            panel = new ObjectPanel(endpoint, param.getContainedTypeName(), params);
                            fillPanelWithValues();
                        }
                    });
                    return;
                }
                if (param instanceof ListParam) {
					frame.invokeLater(new Runnable() {
						@Override
						public void run() {
							panel = new ListPanel(endpoint);
							fillPanelWithValues();
						}
					});
					return;
                    //return panel = new ListPanel(browser, access);
                }
                assert false;
            }
        });
	}

    public String getTooltip() {
        assert frame.isActive();
        return tooltip;
    }




    /*public void subscribe(final EventParam eventParam) {
        assert browser.isActive();
        if (hasSubscribed(eventParam)) {
            LOGGER.error(eventParam + " is already subscribed for " + this);
            return;
        }
        Node node = getNode();
        node.getConnection().subscribe(node.getPath() + "/" + eventParam.getId(), new SubscriptionCallback() {
            @Override
            public EventCallback subscribed(final Subscription subscription) {
                if (subscription == null) {
                    LOGGER.error("subscription failed");
                    return null;
                }
                if (hasSubscribed(eventParam)) {
                    //abort subscription
                    return null;
                }
                userSubscriptions.put(eventParam, subscription);
                LOGGER.debug("subscription succeeded: " + subscription);
                subscription.setDispatcher(browser);
                return new EventCallback() {
                    @Override
                    public void call(SourceInterface content) {
                        assert browser.isActive();
                        LOGGER.info("event " + subscription + " occurred");
                    }
                };
            }
        });

    }*/

    public boolean hasSubscribed(EventParam param) {
        assert frame.isActive();
        return userSubscriptions.containsKey(param);
    }

    public void unsubscribe(EventParam eventParam) {
        assert frame.isActive();
        if (!hasSubscribed(eventParam)) {
            LOGGER.error("could not unsubscribe from " + eventParam);
            return;
        }
        userSubscriptions.get(eventParam).unsubscribe(new LoggingStateCallback(LOGGER, "unsubscribed " + eventParam));
        userSubscriptions.remove(eventParam);
    }

/*

    @Override
    public void onChildChange(final Child child, ListChange.Action action) {
        assert browser.manager.isActive();

        switch (action) {
            case Remove: {
                Dispatching.invokeDispatch(browser, browser.manager, new Runnable() {
                    @Override
                    public void run() {
                        assert browser.manager.isActive();
                        final TreeNode treeNode = (TreeNode) child.getUserObject();
                        if (treeNode == null) {
                            LOGGER.error("child " + child + " had no tree node attached");
                            return;
                        }
                        browser.invokeLater(new Runnable() {
                            @Override
                            public void run() {
                                assert browser.isActive();
                                TreePath path = browser.startChange();
                                //assert treeNode.getParent() == TreeNode.this;
                                if (treeNode.getParent() != null) {
                                    browser.treeModel.removeNodeFromParent(treeNode);
                                }
                                //remove(treeNode);
                                browser.markNodeChanged(TreeNode.this, path);
                            }
                        });
                    }
                });
                break;
            }
        }

    }
*/

    public String getName() {
        return name;
    }

    public String getIconName() {
        return iconName;
    }

    @Override
    public void onUpgrade(Path path) {
    }

    @Override
    public void onChange(Path path) {
    }

    @Override
    public void onChildChange(Path path, ListChange.Action action) {
    }

    public final Frame getFrame() {
        return frame;
    }

	public boolean changeValue(ValueControl component, Object newValue) {
		LOGGER.debug("changing value of " + component + " to '" + newValue + "'");

		if (changedValues == null) {
			changedValues = new HashMap<ValueControl, Object>();
		}
		changedValues.put(component, newValue);

		return true;
	}

	public Path getInstancePath() {
		assert frame.isActive();
		return path;
	}

	public void applyChanges() {
		assert frame.isActive();
		if (changedValues == null) {
			return;
		}
		final Map<ValueControl, Object> changes = changedValues;
		changedValues = null;
		endpoint.getEndpoint().invokeLater(new Runnable() {
			@Override
			public void run() {
				for (Map.Entry<ValueControl, Object> e : changes.entrySet()) {
					final ValueControl key = e.getKey();
					final Path p = path;
					endpoint.getEndpoint().getInstance().storeValue(p, e.getKey().getParam(), e.getValue(), new StateFunctor() {
						@Override
						public void call(Exception e) {
							changes.remove(key);
							if (!changes.isEmpty()) {
								return;
							}
							LOGGER.debug("applied changes for: " + p);
							frame.invokeLater(new Runnable() {
								@Override
								public void run() {
									fillPanelWithValues();
								}
							});
						}
					});
				}
			}
		});
	}

}
