package com.framsticks.gui;

import com.framsticks.core.Node;
import com.framsticks.core.Path;
import com.framsticks.gui.view.*;
import com.framsticks.gui.view.TreeCellRenderer;
import com.framsticks.util.Dispatcher;
import org.apache.log4j.Logger;

import javax.swing.*;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.*;
import java.awt.*;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.HashMap;
import java.util.Map;

/**
 * @author Piotr Sniegowski
 */
public class Frame extends JFrame implements Dispatcher {

    private static final Logger LOGGER = Logger.getLogger(Frame.class.getName());

    protected final Browser browser;

    protected final Dimension screenDimension = Toolkit.getDefaultToolkit().getScreenSize();


    final protected CardLayout cardPanelLayout = new CardLayout();
    protected final JPanel cardPanel = new JPanel();

    protected JScrollPane treeScrollPane;
    protected JTree tree;
    protected DefaultTreeModel treeModel;
    protected javax.swing.tree.MutableTreeNode rootNode;
    //final Instance instance;
    protected JPanel treePanel;
    protected  JPopupMenu treePopupMenu;
    protected JMenuItem treePopupMenuHeader;

    TreeNode currentlyPoppedTreeNode;
    protected JLabel statusBar;
    protected JPanel mainPanel;
    protected JPanel leftPanel;
    protected JPanel normalWorkPanel;
    protected CardLayout mainPanelLayout;

    protected final Map<BrowserEndpoint, EndpointAtFrame> endpoints = new HashMap<BrowserEndpoint, EndpointAtFrame>();

    public Frame(String title, Browser browser) {
		super(title);
        this.browser = browser;
    }

    public void configure() {

        Container contentPane = this.getContentPane();
        contentPane.setLayout(new BorderLayout());

        treePopupMenu = new JPopupMenu("title");
        treePopupMenu.setName("popup");
        treePopupMenuHeader = new JMenuItem();
        treePopupMenuHeader.setForeground(Color.BLUE);
        treePopupMenu.add(treePopupMenuHeader);
        treePopupMenu.addSeparator();
        //TODO: add to favourites
        //TODO: remove from favourites
        //TODO: open in new window as root
        //TODO: refresh
        //TODO: open in console

        treePopupMenu.add(new JMenuItem("Refresh"));
        treePopupMenu.add(new JMenuItem("Open in new frame as root"));
		addNodeActionToTreePopupMenu("Copy path to clipboard", new NodeAction() {
			@Override
			public void actionPerformed(TreeNode treeNode) {
				Path path = treeNode.getInstancePath();
				StringSelection selection = new StringSelection(path.toString());
				getToolkit().getSystemClipboard().setContents(selection, selection);
			}
		});
        //this.add(createMenuItem("Add to favourites", null));
        //this.add(createMenuItem("Remove from favourites", null));

        treePanel = new JPanel();
        treePanel.setLayout(new BorderLayout());

        treeModel = new DefaultTreeModel(null);

        tree = new JTree(treeModel);
        tree.setRootVisible(false);
        ToolTipManager.sharedInstance().registerComponent(tree);

        tree.addTreeSelectionListener(new TreeSelectionListener() {
            @Override
            public void valueChanged(TreeSelectionEvent e) {
                chooseTreeNode(e.getNewLeadSelectionPath());
            }
        });

        tree.setExpandsSelectedPaths(true);
        tree.setEditable(false);
        tree.setDoubleBuffered(true);
        tree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
        tree.setShowsRootHandles(true);
        tree.setRowHeight(26);
        tree.setDoubleBuffered(true);
        tree.addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(MouseEvent e) {
                assert isActive();
                showPopup(e);
            }

            @Override
            public void mouseReleased(MouseEvent e) {
                assert isActive();
                showPopup(e);
            }
        });
        tree.setCellRenderer(new TreeCellRenderer());

        treeScrollPane = new JScrollPane(tree);
        treeScrollPane.setBorder(BorderFactory.createEmptyBorder());

        treePanel.add(treeScrollPane);

        rootNode = new DefaultMutableTreeNode();
        treeModel.setRoot(rootNode);

        normalWorkPanel = new JPanel();
        normalWorkPanel.setLayout(new BorderLayout());

        mainPanel = new JPanel();
        mainPanelLayout = new CardLayout();
        mainPanel.setLayout(mainPanelLayout);
        mainPanel.add(normalWorkPanel, "browser");

        contentPane.add(createMenu(), BorderLayout.NORTH);
        contentPane.add(mainPanel, BorderLayout.CENTER);
        contentPane.add(statusBar, BorderLayout.SOUTH);

        leftPanel = new JPanel();
        leftPanel.setLayout(new BorderLayout());
        JPanel leftTopPanel = createLeftTopPanel();
        if (leftTopPanel != null) {
            leftPanel.add(leftTopPanel, BorderLayout.PAGE_START);
        }
        leftPanel.add(treePanel, BorderLayout.CENTER);
        leftPanel.setBackground(Color.WHITE);
        leftPanel.setForeground(Color.WHITE);

        JSplitPane split = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, leftPanel, cardPanel);
        split.setPreferredSize(new Dimension(this.browser.getConfig().getInteger("size.width", 1000), this.browser.getConfig().getInteger("size.height", 500)));
        split.setMaximumSize(screenDimension);
        split.setOneTouchExpandable(true);
        split.setDividerLocation(250);
        split.setDividerSize(5);

        normalWorkPanel.add(split);

        //this.setVisible(true);
        mainPanelLayout.show(mainPanel, "browser");

        cardPanel.setLayout(cardPanelLayout);

		this.pack();
		tree.requestFocusInWindow();

    }

    protected JMenuBar createMenu() {
        return new JMenuBar();
    }

    protected JPanel createLeftTopPanel() {
        return null;
    }

    public void repaintTree() {
        //LOGGER.debug("repainting");
        //tree.repaint();
    }

    public EndpointAtFrame getOrCreateEndpointAtFrame(BrowserEndpoint endpoint) {
        assert isActive();
        if (endpoints.containsKey(endpoint)) {
            return endpoints.get(endpoint);
        }
        EndpointAtFrame e = new EndpointAtFrame(endpoint, this);
        endpoints.put(endpoint, e);
        return e;
    }

    public void addRootPath(BrowserEndpoint endpoint, Path path) {
        assert isActive();
        TreeNode node = new TreeNode(this, getOrCreateEndpointAtFrame(endpoint), path);
        treeModel.insertNodeInto(node, rootNode, rootNode.getChildCount());
        tree.expandPath(new TreePath(rootNode));
    }

    @Override
    public boolean isActive() {
        return SwingDispatcher.instance.isActive();
    }

    @Override
    public void invokeLater(Runnable runnable) {
        SwingDispatcher.instance.invokeLater(runnable);
    }

    public Action addActionToTreePopupMenu(Action action) {
        assert isActive();
        treePopupMenu.add(action);
        return action;
    }

    public Action addNodeActionToTreePopupMenu(String title, final NodeAction nodeAction) {
        assert isActive();
        return addActionToTreePopupMenu(new AbstractAction(title) {
            @Override
            public void actionPerformed(ActionEvent e) {
                TreeNode treeNode = getCurrentlyPoppedTreeNode();
                if (treeNode == null) {
                    return;
                }
                nodeAction.actionPerformed(treeNode);
            }
        });
    }


    public void showPanel(Panel panel) {
        assert isActive();
        assert panel != null;
        cardPanelLayout.show(cardPanel, panel.getFullName());
    }

    private void showPopup(MouseEvent e) {
        assert isActive();
        if (!e.isPopupTrigger()) {
            return;
        }
        currentlyPoppedTreeNode = findTreeNodeByTreePath(tree.getPathForLocation(e.getX(), e.getY()));
        if (currentlyPoppedTreeNode == null) {
            return;
        }
        treePopupMenu.show(e.getComponent(), e.getX(), e.getY());
        //currentlyPoppedPanel.getNode().getFramsClass().getName()
        //treePopupMenuHeader.setText(currentlyPoppedTreeNode.getNode().getPath());
    }

    public TreeNode getCurrentlyPoppedTreeNode() {
        assert isActive();
        return currentlyPoppedTreeNode;
    }


    public void clear() {
        treeModel.setRoot(null);
        cardPanel.removeAll();
        cardPanel.updateUI();
        tree.setEnabled(false);
    }


    public void markNodeChanged(MutableTreeNode treeNode, TreePath path) {
        assert isActive();
        treeModel.nodeStructureChanged(treeNode);
        tree.setSelectionPath(path);
        tree.repaint();
    }

    public TreePath startChange() {
        return tree.getSelectionPath();
    }


    public void selectTreeNode(final TreeNode treeNode) {
        assert isActive();
/*		final Panel panel = treeNode.getOrCreatePanel();
		if (panel == null) {
			return;
		}
		panel.setCurrentTreeNode(treeNode);
		treeNode.updateData();
		showPanel(panel);*/
    }

    public TreeNode findTreeNodeByTreePath(TreePath treePath) {
        assert isActive();
        if (treePath == null) {
            return null;
        }
        if (!(treePath.getLastPathComponent() instanceof TreeNode)) {
            return null;
        }
        return (TreeNode)treePath.getLastPathComponent();
    }



    public void chooseTreeNode(TreePath treePath) {
        assert isActive();
        final TreeNode treeNode = findTreeNodeByTreePath(treePath);
        if (treeNode == null) {
            return;
        }
        treeNode.select();
    }


}
