// This file is a part of Framsticks SDK.  http://www.framsticks.com/
// Copyright (C) 1999-2015  Maciej Komosinski and Szymon Ulatowski.
// See LICENSE.txt for details.

#include <frams/param/multiparamload.h>
#include <frams/virtfile/stdiofile.h>
#include <frams/errmgr/stdouterr.h>

/**
 @file
 Sample code: Loading Framsticks "objects" (structures).

 After loading, objects are printed ("saved") to standard output.
 Additional information and messages are printed to standard error.
 You can redirect both streams if needed.

 \include loader_test_param.cpp
 */

struct Data
{
public:
	SString text, longtext, deftext;
	int i1, i2, i3;
	float f1, f2, f3;
	ExtValue x;
	int notchanged,notloaded,notsaved;
};

#define FIELDSTRUCT Data
ParamEntry data_paramtab[] =
{
	{ "Data", 1, 12, "data", },
	{ "text", 0, 0, "Text", "s 0 10", FIELD(text), }, // 10 = length limit
	{ "longtext", 0, 0, "Long text", "s 1", FIELD(longtext), }, // 1 = multiline, 0 = unlimited
	{ "deftext", 0, 0, "Text with default value", "s 0 0 some text", FIELD(deftext), },
	{ "i1", 0, 0, "Integer 1", "d", FIELD(i1), },          // unrestricted integer
	{ "i2", 0, 0, "Integer 2", "d -5 5 1", FIELD(i2), }, // [-5..5] integer, default=1
	{ "i3", 0, 0, "Integer 3", "d -1 3 0 ~Minus~Zero~One~Two~Three", FIELD(i3), }, // [0..3] + text labels (Param::getText())
	{ "f1", 0, 0, "Float 1", "d", FIELD(f1), },          // unrestricted float
	{ "f2", 0, 0, "Float 2", "f -100 100 -100", FIELD(f2), }, // [-100..100] float, default=-100
	{ "x", 0, 0, "Untyped", "x", FIELD(x), }, // any type (class ExtValue)
	{ "notchanged", 0, PARAM_READONLY, "Read only field", "d", FIELD(notchanged), }, // neither load() nor setDefault() can change this
	{ "notloaded", 0, PARAM_DONTLOAD, "Unloadable field", "d", FIELD(notloaded), }, // load() does not change this
	{ "notsaved", 0, PARAM_DONTSAVE, "Unsaveable field", "d", FIELD(notsaved), }, // save() skips this field
	{ 0, 0, 0, },
};
#undef FIELDSTRUCT

int main(int argc, char*argv[])
{
	if (argc < 2)
	{
		fprintf(stderr, "Arguments: filename\n");
		return 1;
	}

	StdioFILEDontClose virt_stderr(stderr);
	StdioFILEDontClose virt_stdout(stdout);
	StdoutErrorHandler error_handler(0, &virt_stderr);
	StdioFileSystem_autoselect stdiofilesys;
	MultiParamLoader loader(argv[1]);

	Data data;
	Param param(data_paramtab, &data);

	data.notchanged=100;
	data.notloaded=200;

	loader.addObject(&param);
	loader.breakOn(MultiParamLoader::OnError + MultiParamLoader::BeforeObject + MultiParamLoader::AfterObject + MultiParamLoader::OnComment);

	while (int status = loader.go())
	{
		switch (status)
		{
		case MultiParamLoader::OnComment:
			fprintf(stderr, "comment: %s\n", (const char*)loader.getComment());
			break;

		case MultiParamLoader::AfterObject:
			fprintf(stderr, "loaded:\n");
			for (int i = 0; i < param.getPropCount(); i++)
				fprintf(stderr, "%s=%s\n", param.id(i), (const char*)param.getText(i));
			fprintf(stderr, "type of 'x' is: %s\n", (const char*)data.x.typeDescription());
			fprintf(stderr, "-----\n\n");
			param.save(&virt_stdout);
			break;

		case MultiParamLoader::BeforeObject:
			fprintf(stderr, "----- object found, loading...\n");
			data.notchanged++;
			param.setDefault(); //reset (initialize) struct contents
			break;

		case MultiParamLoader::OnError:
			fprintf(stderr, "Error: %s", (const char*)loader.getError());
		}
	}
	return 0;
}
