package com.framsticks.params;

import static org.fest.assertions.Assertions.*;

import java.util.Arrays;
import java.util.Map;
import java.util.TreeMap;

import org.testng.annotations.DataProvider;
import org.testng.annotations.Test;

import com.framsticks.params.AccessOperations;
import com.framsticks.params.FramsClass;
import com.framsticks.params.ListSink;
import com.framsticks.params.ReflectionAccess;
import com.framsticks.params.types.UniversalParam;
import com.framsticks.parsers.Savers;
import com.framsticks.test.TestConfiguration;
import com.framsticks.test.TestSerializedClass;
import com.framsticks.util.lang.Containers;
import com.framsticks.util.lang.Pair;


public class ParamsUtilTest extends TestConfiguration {

	@Test
	public void testVariousSerializations() {
		FramsClass framsClass;
		ReflectionAccess access;

		framsClass = FramsClass.build().forClass(TestSerializedClass.class);
		assertThat(framsClass.getParamCount()).isEqualTo(1);
		assertThat(framsClass.getParam("theArray")).isInstanceOf(UniversalParam.class);

		access = new ReflectionAccess(TestSerializedClass.class, framsClass);

		assertThat(Savers.saveFramsClass(new ListSink(), framsClass).getOut()).isEqualTo(
				Arrays.asList(
					"class:",
					"name:TestSerializedClass",
					"id:TestSerializedClass",
					"",
					"prop:",
					"id:theArray",
					"name:TheArray",
					"type:x",
					""
				)
			);

		TestSerializedClass test = new TestSerializedClass();
		test.getTheArray().add(1);
		test.getTheArray().add(3.0);
		test.getTheArray().add("word");

		ListSink sink = new ListSink();
		AccessOperations.savePrimitives(access.select(test), sink);
		assertThat(sink.getOut()).isEqualTo(Arrays.asList(
				"TestSerializedClass:",
				"theArray:@Serialized:[1,3.0,\"word\"]",
				""
			));



		Map<String, Object> f12 = new TreeMap<String, Object>();
		f12.put("a", 123);
		f12.put("b", Arrays.asList(7, 8, 9));
		assertThat(ParamsUtil.serialize(f12)).isEqualTo("@Serialized:{\"a\":123,\"b\":[7,8,9]}");

		// List<Object> f18 = new ArrayList<>();
		// List<Object> f18_0 = new ArrayList<>();
		// List<Object> f18_1 = Arrays.asList("x", f18);
		// List<Object> f18_2 = new ArrayList<>();
		// f18_2.add(f18_0);
		// f18_0.add(f18_2);
		// f18.add(f18_0);
		// f18.add(f18_1);
		// f18.add(f18_2);

		// root[0] is f18_0
		// root[0]==root[2][0], root[1][1]==root, root[1][1][1][1][0]=="x".

		// assertThat(ParamsUtil.serialize(f18)).isEqualTo("@Serialized:[[^3],[\"x\",^0],[^1]]");

	}


	@Test(dependsOnMethods = "testVariousSerializations", dataProvider = "serializationData")
	public void testSerialization(Object object, String string) {
		assertThat(ParamsUtil.serialize(object)).isEqualTo(string);
	}

	@Test(dependsOnMethods = "testSerialization", dataProvider = "serializationData")
	public void testDeserialization(Object object, String string) {
		assertThat(ParamsUtil.deserialize(string, Object.class)).isEqualTo(object);
	}

	@DataProvider
	public Object[][] serializationData() {
		Object array = Arrays.asList(1, 2);

		return new Object[][] {
			{ 2, "2" },
			{ 0.5, "0.5" },
			{ "xx", "xx" },
			{ "@Serialized:", "@Serialized:\"@Serialized:\"" },
			{ Arrays.asList(12, null, "abc"), "@Serialized:[12,null,\"abc\"]" },
			{ Arrays.asList(Arrays.asList(Arrays.asList())), "@Serialized:[[[]]]" },
			{ Arrays.asList(1, Containers.buildMap(Pair.make("a", 2), Pair.make("b", "xx"), Pair.make("c", null)), "y"), "@Serialized:[1,{\"a\":2,\"b\":\"xx\",\"c\":null},\"y\"]" },
			{ new OpaqueObject("Population", 0xaabbccddL), "@Serialized:Population<0xaabbccdd>" },
			{ Arrays.asList("x", new OpaqueObject("Population", 0xaabbccddL)), "@Serialized:[\"x\",Population<0xaabbccdd>]" },
			{ Containers.buildMap(Pair.make("a", array), Pair.make("b", array)), "@Serialized:{\"a\":[1,2],\"b\":^1}"}
		};
	}

}
