package com.framsticks.experiment;


import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.framsticks.params.EventListener;
import com.framsticks.params.MessageLogger;
import com.framsticks.params.annotations.FramsClassAnnotation;
import com.framsticks.params.annotations.ParamAnnotation;
import com.framsticks.structure.messages.ListChange;
import com.framsticks.structure.messages.Message;
import com.framsticks.util.dispatching.FutureHandler;
import com.framsticks.util.dispatching.Future;

@FramsClassAnnotation
public abstract class NetLoadSaveLogic<NF extends NetFile> extends AbstractExperimentLogic {
	private static final Logger log = LogManager.getLogger(NetLoadSaveLogic.class);


	protected final Class<NF> netJavaClass;

	protected final MessageLogger messages = new MessageLogger(NetLoadSaveLogic.class);

	/**
	 * @param experiment
	 */
	public NetLoadSaveLogic(Experiment parentExperiment, Class<NF> netJavaClassArg) {
		super(parentExperiment);
		this.netJavaClass = netJavaClassArg;

		experiment.addSimulatorsListener(new EventListener<ListChange>() {

			@Override
			public void action(final ListChange change) {
				assert experiment.isActive();
				final Simulator simulator = experiment.getSimulators().get(change.getIdentifier());
				log.debug("processing list change: {}", change);

				if (change.getAction() == ListChange.Action.Add) {
					log.debug("registering in {}", simulator);
					simulator.getRemoteTree().getRegistry().registerAndBuild(netJavaClass);
				}

				if (!change.hasHint("stoped")) {
					issueNetloadIfReady(change, simulator);
					return;
				}

				log.debug("issuing netsave to: {}", simulator);
				simulator.netsave(netJavaClass, new Future<NF>(simulator) {

					@Override
					protected void result(NF net) {
						log.debug("netsave of {} done: {}", simulator, net.getShortDescription());
						netsave(simulator, net);
						issueNetloadIfReady(change, simulator);
					}
				});
			}
		});
	}

	protected void issueNetloadIfReady(ListChange change, final Simulator simulator) {
		if (!change.hasHint("ready")) {
			return;
		}
		log.debug("issuing netload to: {}", simulator);
		netload(simulator, new Future<NF>(simulator) {

			@Override
			protected void result(final NF net) {
				if (net == null) {
					log.debug("no file for upload provided - leaving simulator idle");
					return;
				}

				simulator.netload(net, new Future<Object>(this) {

					@Override
					protected void result(Object result) {
						NetLoadSaveLogic.this.messages.info("netload", "done " + net.getShortDescription());
						log.debug("netload of {} done", net.getShortDescription());
						simulator.start();
					}
				});
			}
		});
	}

	public abstract void netload(Simulator simulator, FutureHandler<NF> net);

	public abstract void netsave(Simulator simulator, NF net);

	/**
	 * @return the netJavaClass
	 */
	@ParamAnnotation(name = "Java class representing netfile")
	public String getNetJavaClassName() {
		return netJavaClass.getCanonicalName();
	}

	@ParamAnnotation(id = "messages")
	public void addMessageListener(EventListener<Message> listener) {
		messages.add(listener);
	}

	@ParamAnnotation(id = "messages")
	public void removeMessageListener(EventListener<Message> listener) {
		messages.remove(listener);
	}

}
