package com.framsticks.experiment;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.framsticks.params.EventListener;
import com.framsticks.params.MessageLogger;
import com.framsticks.params.ParamFlags;
import com.framsticks.params.SimpleUniqueList;
import com.framsticks.params.annotations.AutoAppendAnnotation;
import com.framsticks.params.annotations.FramsClassAnnotation;
import com.framsticks.params.annotations.ParamAnnotation;
import com.framsticks.remote.RemoteTree;
import com.framsticks.structure.Path;
import com.framsticks.structure.messages.ListChange;
import com.framsticks.structure.messages.Message;
import com.framsticks.util.ExceptionHandler;
import com.framsticks.util.FramsticksException;
import com.framsticks.util.dispatching.AbstractJoinable;
import com.framsticks.util.dispatching.BufferedDispatcher;
import com.framsticks.util.dispatching.Dispatcher;
import com.framsticks.util.dispatching.DispatcherSetable;
import com.framsticks.util.dispatching.Dispatching;
import com.framsticks.util.dispatching.Future;
import com.framsticks.util.dispatching.FutureHandler;
import com.framsticks.util.dispatching.Joinable;
import com.framsticks.util.dispatching.JoinableCollection;
import com.framsticks.util.dispatching.JoinableParent;
import com.framsticks.util.dispatching.JoinableState;
import com.framsticks.util.dispatching.RunAt;

@FramsClassAnnotation
public class Experiment extends AbstractJoinable implements Dispatcher<Experiment>, DispatcherSetable<Experiment>, JoinableParent, ExceptionHandler {
	private static final Logger log = LogManager.getLogger(Experiment.class);

	protected final JoinableCollection<Simulator> simulatorAsJoinables = new JoinableCollection<Simulator>().setObservableName("simulators");

	protected final JoinableCollection<RemoteTree> simulatorCandidates = new JoinableCollection<RemoteTree>().setObservableName("candidates");

	protected final SimpleUniqueList<Simulator> simulators = new SimpleUniqueList<>(Simulator.class, 's');

	protected final SimpleUniqueList<Simulator> oldSimulators = new SimpleUniqueList<>(Simulator.class, 's');

	protected final BufferedDispatcher<Experiment> bufferedDispatcher = new BufferedDispatcher<>(this);

	protected SimulatorProvider simulatorProvider;

	protected String expdef;

	protected final MessageLogger messages = new MessageLogger(NetLoadSaveLogic.class);

	/**
	 *
	 */
	public Experiment() {
		super();
		bufferedDispatcher.setBuffer(false);

		Dispatching.dispatchLog(this, log, Level.DEBUG, "first task");
	}

	/**
	 * @return the simulatorCandidates
	 */
	public JoinableCollection<RemoteTree> getSimulatorCandidates() {
		return simulatorCandidates;
	}

	@ParamAnnotation
	public Map<String, Simulator> getSimulators() {
		return simulators.getView();
	}

	@ParamAnnotation(id = "old_simulators")
	public Map<String, Simulator> getOldSimulators() {
		return oldSimulators.getView();
	}

	/**
	 * @return the dispatcher
	 */
	@Override
	public Dispatcher<Experiment> getDispatcher() {
		return bufferedDispatcher;
	}

	/**
	 * @param dispatcher the dispatcher to set
	 */
	@Override
	public void setDispatcher(Dispatcher<Experiment> dispatcher) {
		bufferedDispatcher.setTargetDispatcher(dispatcher);
	}

	/**
	 * @return the simulatorProvider
	 */
	@ParamAnnotation(flags = ParamFlags.USERREADONLY)
	public SimulatorProvider getSimulatorProvider() {
		return simulatorProvider;
	}

	/**
	 * @param simulatorProvider the simulatorProvider to set
	 */
	@AutoAppendAnnotation
	@ParamAnnotation
	public void setSimulatorProvider(SimulatorProvider simulatorProvider) {
		this.simulatorProvider = simulatorProvider;
	}

	/**
	 * @return the expdef
	 */
	@ParamAnnotation(flags = ParamFlags.USERREADONLY)
	public String getExpdef() {
		return expdef;
	}

	/**
	 * @param expdef the expdef to set
	 */
	@ParamAnnotation
	public void setExpdef(String expdef) {
		this.expdef = expdef;
	}


	@ParamAnnotation(id = "simulators_changed")
	public void addSimulatorsListener(EventListener<ListChange> listener) {
		simulators.addListener(listener);
	}

	@ParamAnnotation(id = "simulators_changed")
	public void removeSimulatorsListener(EventListener<ListChange> listener) {
		simulators.removeListener(listener);
	}

	@AutoAppendAnnotation
	public void addSimulator(Simulator simulator) {
		log.debug("add simulator {}", simulator);
		simulators.add(simulator);
		simulatorAsJoinables.add(simulator);
		simulators.fireChildrenChange(simulator, ListChange.Action.Modify, "ready");

		simulatorCandidates.remove(simulator.getRemoteTree());
	}

	protected void removeSimulator(Simulator simulator) {
		simulatorAsJoinables.remove(simulator);
		simulators.remove(simulator);
		oldSimulators.add(simulator);
	}

	@ParamAnnotation(id = "old_simulators_changed")
	public void addOldSimulatorsListener(EventListener<ListChange> listener) {
		oldSimulators.addListener(listener);
	}

	@ParamAnnotation(id = "old_simulators_changed")
	public void removeOldSimulatorsListener(EventListener<ListChange> listener) {
		oldSimulators.removeListener(listener);
	}

	@Override
	public String getName() {
		return "experiment";
	}

	@Override
	public void childChangedState(Joinable joinable, JoinableState state) {
		proceedToState(state);
	}

	@Override
	protected void joinableStart() {
		bufferedDispatcher.createThreadIfNeeded();
		Dispatching.use(bufferedDispatcher, this);

		Dispatching.use(simulatorAsJoinables, this);
		Dispatching.use(simulatorProvider, this);
		Dispatching.use(simulatorCandidates, this);

		tryProvideAllSimulators(Future.<List<Simulator>>doNothing(this));
	}

	@Override
	protected void joinableInterrupt() {

		Dispatching.drop(simulatorAsJoinables, this);
		Dispatching.drop(simulatorProvider, this);
		Dispatching.drop(simulatorCandidates, this);

		Dispatching.drop(bufferedDispatcher, this);

		finishJoinable();
	}

	@Override
	protected void joinableFinish() {
		log.debug("finishing experiment {}", this);
	}

	@Override
	protected void joinableJoin() throws InterruptedException {

		Dispatching.join(simulatorAsJoinables);
		Dispatching.join(simulatorProvider);
		Dispatching.join(simulatorCandidates);
		// Dispatching.join(bufferedDispatcher.getTargetDispatcher());
	}

	@Override
	public void handle(FramsticksException exception) {
		log.error("caught exception: ", exception);
	}

	@Override
	public boolean isActive() {
		return bufferedDispatcher.isActive();
	}

	@Override
	public void dispatch(RunAt<? extends Experiment> runnable) {
		bufferedDispatcher.dispatch(runnable);
	}

	// @ParamAnnotation(paramType = ProcedureParam.class)
	// public void connectToSimulator(String address) {
	//	SimulatorConnector connector = new SimulatorConnector();
	//	connector.setAddress(address);
	//	connector.attachTo(this);
	// }
	//

	public Simulator createSimulator(RemoteTree tree, Path path) {
		return new Simulator(this, tree, path);
	}

	public void tryProvideNextSimulator(final FutureHandler<Simulator> future) {
		log.debug("trying to provide next simulator");
		simulatorProvider.provideSimulator(new SimulatorSpecification(this, expdef), new Future<Simulator>(future) {

			@Override
			protected void result(Simulator result) {
				assert isActive();
				if (result != null) {
					addSimulator(result);
				} else {
					log.debug("no more simulators remaining");
				}
				future.pass(result);

			}
		});

	}

	public void tryProvideAllSimulators(final FutureHandler<List<Simulator>> future) {
		log.debug("trying to provide all simulators");
		final List<Simulator> list = new LinkedList<>();

		tryProvideNextSimulator(new Future<Simulator>(future) {

			@Override
			protected void result(Simulator result) {
				if (result == null) {
					future.pass(list);
					return;
				}
				list.add(result);
				tryProvideNextSimulator(this);
			}
		});

	}

	@ParamAnnotation(id = "messages")
	public void addMessageListener(EventListener<Message> listener) {
		messages.add(listener);
	}

	@ParamAnnotation(id = "messages")
	public void removeMessageListener(EventListener<Message> listener) {
		messages.remove(listener);
	}

}
