package com.framsticks.framclipse.editors.configuration;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextHover;
import org.eclipse.jface.text.TextAttribute;
import org.eclipse.jface.text.contentassist.ContentAssistant;
import org.eclipse.jface.text.contentassist.IContentAssistant;
import org.eclipse.jface.text.presentation.IPresentationReconciler;
import org.eclipse.jface.text.presentation.PresentationReconciler;
import org.eclipse.jface.text.reconciler.IReconciler;
import org.eclipse.jface.text.reconciler.MonoReconciler;
import org.eclipse.jface.text.rules.BufferedRuleBasedScanner;
import org.eclipse.jface.text.rules.DefaultDamagerRepairer;
import org.eclipse.jface.text.rules.ITokenScanner;
import org.eclipse.jface.text.rules.Token;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jface.text.source.SourceViewerConfiguration;
import org.eclipse.swt.graphics.Color;
import org.jdom.Attribute;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.xpath.XPath;

import com.framsticks.framclipse.Framclipse;
import com.framsticks.framclipse.editors.EditorType;
import com.framsticks.framclipse.editors.FramclipseEditor;
import com.framsticks.framclipse.editors.codeCompletion.FramscriptCompletionProcessor;
import com.framsticks.framclipse.editors.codeCompletion.PropertyCompletionProcessor;
import com.framsticks.framclipse.editors.codeCompletion.SyntaxUtils;
import com.framsticks.framclipse.editors.hover.FramscriptTextHover;
import com.framsticks.framclipse.syntaxColoring.ColorManager;
import com.framsticks.framclipse.syntaxColoring.FramclipseColors;
import com.framsticks.framclipse.syntaxColoring.FramclipseCommentScanner;
import com.framsticks.framclipse.syntaxColoring.FramscriptCodeScanner;


public class FramscriptSourceViewerConfiguration extends SourceViewerConfiguration {

	private FramscriptCodeScanner codeScanner;
	private final EditorType editorType;
	private FramclipseEditor editor;

	public FramscriptSourceViewerConfiguration(EditorType editorType, FramclipseEditor editor) {
		System.out.println("creating sourceViewerConfiguration for type: " + editorType);
		this.editorType = editorType;
		this.editor = editor;
	}

	// /////////////////////////////////////////////////////////////////////////
	// code highlighting
	// ///////////////////////////////////////////////////////////////////////

	@Override
	public String[] getConfiguredContentTypes(ISourceViewer sourceViewer) {
		return new String[] { IDocument.DEFAULT_CONTENT_TYPE, FramscriptPartitionScanner.CODE };
	}

	protected ITokenScanner getFramscriptCodeScanner() {
		ColorManager cm = Framclipse.getDefault().getColorManager();
		if (codeScanner == null) {
			codeScanner = new FramscriptCodeScanner();
			Color defaultColor = cm.getColor(FramclipseColors.DEFAULT);
			TextAttribute textAttribute = new TextAttribute(defaultColor);
			Token defaultToken = new Token(textAttribute);
			codeScanner.setDefaultReturnToken(defaultToken);
		}
		return codeScanner;
	}
	
	private ITokenScanner commentScanner = null;
	
	protected ITokenScanner getCommentScanner()
	{
		if(commentScanner == null)
			commentScanner = new FramclipseCommentScanner();
		
		return commentScanner;
	}

	@Override
	public String getConfiguredDocumentPartitioning(ISourceViewer sourceViewer) {
		return Framclipse.FRAMSCRIPT_PARTITIONING;
	}

	@Override
	public IPresentationReconciler getPresentationReconciler(ISourceViewer sourceViewer) {
		PresentationReconciler reconciler = new PresentationReconciler();
		String partitioning = getConfiguredDocumentPartitioning(sourceViewer);
		reconciler.setDocumentPartitioning(partitioning);

		DefaultDamagerRepairer dr = new DefaultDamagerRepairer(getFramscriptCodeScanner());
		reconciler.setDamager(dr, FramscriptPartitionScanner.CODE);
		reconciler.setRepairer(dr, FramscriptPartitionScanner.CODE);

		ColorManager colorManager = Framclipse.getDefault().getColorManager();
		Color color = colorManager.getColor(FramclipseColors.DESCRIPTION);
		dr = new DefaultDamagerRepairer(new SingleTokenScanner(new TextAttribute(color)));
		reconciler.setDamager(dr, FramscriptPartitionScanner.DESCRIPTION);
		reconciler.setRepairer(dr, FramscriptPartitionScanner.DESCRIPTION);

		color = colorManager.getColor(FramclipseColors.DEFAULT);
		dr = new DefaultDamagerRepairer(new SingleTokenScanner(new TextAttribute(color)));
		reconciler.setDamager(dr, FramscriptPartitionScanner.UNDEFINED);
		reconciler.setRepairer(dr, FramscriptPartitionScanner.UNDEFINED);
		reconciler.setDamager(dr, IDocument.DEFAULT_CONTENT_TYPE);
		reconciler.setRepairer(dr, IDocument.DEFAULT_CONTENT_TYPE);
		
		//colorManager = Framclipse.getDefault().getColorManager();
		//color = colorManager.getColor(FramclipseColors.COMMENT);
		//dr = new DefaultDamagerRepairer(new SingleTokenScanner(new TextAttribute(color)));
		dr = new DefaultDamagerRepairer(getCommentScanner());
		reconciler.setDamager(dr, FramscriptPartitionScanner.COMMENT);
		reconciler.setRepairer(dr, FramscriptPartitionScanner.COMMENT);

		return reconciler;
	}

	static class SingleTokenScanner extends BufferedRuleBasedScanner {
		public SingleTokenScanner(TextAttribute attribute) {
			setDefaultReturnToken(new Token(attribute));
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// content assist
	// ///////////////////////////////////////////////////////////////////////

	@SuppressWarnings("unchecked")
	@Override
	public IContentAssistant getContentAssistant(ISourceViewer sourceViewer) {

		ContentAssistant assistant = new ContentAssistant();
		String partitioning = getConfiguredDocumentPartitioning(sourceViewer);
		assistant.setDocumentPartitioning(partitioning);

		Set<String> contexts = new HashSet<String>();
		Document framscriptContext = Framclipse.getDefault().getFramscriptContext();
		try {
			String query = "//file[@pattern='*." + editorType.getExtension() + "']/code/context";
			XPath xpath = XPath.newInstance(query);
			List<Element> list = xpath.selectNodes(framscriptContext);
			for (Element element : list) {
				Attribute name = element.getAttribute(SyntaxUtils.NAME_ATTRIBUTE);
				contexts.add(name.getValue());
			}
		} catch (Exception e) {
			e.printStackTrace();
		}

		FramscriptCompletionProcessor framscriptCompletionProcessor = new FramscriptCompletionProcessor(
				contexts,editor,editorType);
		assistant.setContentAssistProcessor(framscriptCompletionProcessor,
				FramscriptPartitionScanner.CODE);

		PropertyCompletionProcessor propertyCompletionProcessor = getPropertyCompletionProcessor(
				contexts, framscriptContext);
		assistant.setContentAssistProcessor(propertyCompletionProcessor,
				IDocument.DEFAULT_CONTENT_TYPE);

		assistant.enableAutoActivation(true);
		assistant.setAutoActivationDelay(500);
		int orientation = IContentAssistant.PROPOSAL_OVERLAY;
		assistant.setProposalPopupOrientation(orientation);
		orientation = IContentAssistant.CONTEXT_INFO_ABOVE;
		assistant.setContextInformationPopupOrientation(orientation);

		return assistant;
	}

	@SuppressWarnings("unchecked")
	private PropertyCompletionProcessor getPropertyCompletionProcessor(Set<String> contexts,
			Document framscriptContext) {

		Map<String, String> propertyTypes = new HashMap<String, String>();
		try {
			String query = "//file[@pattern='*." + editorType.getExtension() + "']/property";
			XPath xpath = XPath.newInstance(query);
			List<Element> list = xpath.selectNodes(framscriptContext);
			for (Element element : list) {
				Attribute name = element.getAttribute(SyntaxUtils.NAME_ATTRIBUTE);
				Attribute type = element.getAttribute(SyntaxUtils.TYPE_ATTRIBUTE);
				propertyTypes.put(type.getValue(), name.getValue());
			}
		} catch (Exception e) {
			e.printStackTrace();
		}

		Set<String> objectContexts = new HashSet<String>();
		try {
			String query = "//file[@pattern='*." + editorType.getExtension() + "']/object";
			XPath xpath = XPath.newInstance(query);
			List<Element> list = xpath.selectNodes(framscriptContext);
			for (Element element : list) {
				Attribute context = element.getAttribute(SyntaxUtils.CONTEXT_ATTRIBUTE);
				objectContexts.add(context.getValue());
			}
		} catch (Exception e) {
			e.printStackTrace();
		}

		PropertyCompletionProcessor propertyCompletionProcessor = new PropertyCompletionProcessor(
				contexts, objectContexts, propertyTypes);
		return propertyCompletionProcessor;
	}

	// /////////////////////////////////////////////////////////////////////////
	// text hover
	// ///////////////////////////////////////////////////////////////////////

	@Override
	public ITextHover getTextHover(ISourceViewer sourceViewer, String contentType) {
		if (FramscriptPartitionScanner.CODE.equals(contentType)) {
			return new FramscriptTextHover();
		}
		return null;
	}

	@Override
	public IReconciler getReconciler(ISourceViewer sourceViewer) {
		FramclipseReconcilingStrategy strategy = new FramclipseReconcilingStrategy();
		strategy.setEditor(editor);
		
		MonoReconciler reconciler = new MonoReconciler(strategy,false);
		
        
        return reconciler;
	}
	
	

}
