package com.framsticks.framclipse.editors.codeCompletion;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.xpath.XPath;

import com.framsticks.framclipse.Framclipse;


/**
 * Utils for finding function and field names in document and describing them
 * (e.g. using type).
 * 
 * @author Jan Kuniak
 */
public class SyntaxUtils {

	public static final String ROOT = "/framscript";

	public static final String CONTEXT_ATTRIBUTE = "context";
	public static final String FUNCTION_ATTRIBUTE = "function";
	public static final String NAME_ATTRIBUTE = "name";
	public static final String TYPE_ATTRIBUTE = "type";

	public static final String ARGUMENT_ELEMENT = "argument";
	public static final String ARGUMENTS_ELEMENT = "arguments";
	public static final String DESCRIPTION_ELEMENT = "description";
	public static final String ELEMENT_ELEMENT = "element";
	public static final String TYPE_ELEMENT = "type";

	// /////////////////////////////////////////////////////////////////////////
	// Before
	// ///////////////////////////////////////////////////////////////////////

	/**
	 * Returns identifier before the cursor (if any).
	 * 
	 * @param document
	 *            document in which element is to be found.
	 * @param offset
	 *            cursor position in the document.
	 * @return identifier before the cursor if any, "" if not found.
	 */
	public static String getElementBefore(IDocument document, int offset) {
		String result = "";
		int i = offset - 1;
		try {
			while ((i > 0)) {
				char ch = document.getChar(i);
				if (!Character.isJavaIdentifierPart(ch)) {
					return result;
				}

				result = ch + result;
				i--;
			}
		} catch (BadLocationException e) {
			e.printStackTrace();
		}
		return result;
	}

	/**
	 * Returns brace block, starting with opening brace, ending with closing
	 * brace and regarding braces inside.
	 * 
	 * @param document
	 *            document to search brace block in.
	 * @param offset
	 *            cursor position in the document.
	 * @return brace block if found, "" otherwise.
	 */
	public static String getBraceBlockBefore(IDocument document, int offset) {
		String result = "";
		int i = offset - 1;
		int bracketsCount = 0;
		try {
			while ((i > 0)) {
				char ch = document.getChar(i);

				switch (ch) {
				case ')':
					bracketsCount++;
					break;
				case '(':
					bracketsCount--;
					break;
				case '\n':
					return result;
				default:
					if (bracketsCount == 0) {
						return result;
					}
				}
				result = ch + result;
				i--;
			}
		} catch (BadLocationException e) {
			e.printStackTrace();
		}
		return result;
	}

	/**
	 * Returns list of elements (Classes, methods, fields) connected with dots
	 * ending with the element before the cursor position. Functions does not
	 * contain braces blocks.
	 * 
	 * @param document
	 *            document to search elements in.
	 * @param offset
	 *            cursor position in the document.
	 * @return list of strings found, may be empty, never null.
	 */
	public static List<String> getElementsBefore(IDocument document, int offset) {

		List<String> objects = new ArrayList<String>();
		String object = "";
		try {
			char probablyDot = document.getChar(offset);

			while (probablyDot == '.') {
				String bracketBlock = SyntaxUtils.getBraceBlockBefore(document, offset);
				offset -= bracketBlock.length();
				object = SyntaxUtils.getElementBefore(document, offset);
				if (object.length() == 0) {
					return objects;
				}
				objects.add(object);

				offset -= object.length() + 1;
				if (offset < 0) {
					return objects;
				}

				probablyDot = document.getChar(offset);
			}
		} catch (BadLocationException e) {
			e.printStackTrace();
		}

		return objects;
	}

	// /////////////////////////////////////////////////////////////////////////
	// After
	// ///////////////////////////////////////////////////////////////////////

	/**
	 * Returns identifier after the cursor position (if any) including character
	 * at position.
	 * 
	 * @param document
	 *            document in which element is to be found.
	 * @param offset
	 *            cursor position in the document.
	 * @return identifier after the cursor if any, "" if not found.
	 */
	public static String getElementAfter(IDocument document, int offset) {
		String result = "";
		int i = offset;
		try {
			while ((i < document.getLength())) {
				char ch = document.getChar(i);
				if (!Character.isJavaIdentifierPart(ch)) {
					return result;
				}

				result += ch;
				i++;
			}
		} catch (BadLocationException e) {
			e.printStackTrace();
		}
		return result;
	}

	// /////////////////////////////////////////////////////////////////////////
	// At
	// ///////////////////////////////////////////////////////////////////////

	/**
	 * Returns list of elements (Classes, methods, fields) connected with dots
	 * ending with the element at the cursor position. Functions does not
	 * contain braces blocks.
	 * 
	 * @param document
	 *            document to search elements in.
	 * @param offset
	 *            cursor position in the document.
	 * @return list of strings found, may be empty, never null.
	 */
	public static String getElementAt(IDocument document, int offset) {
		return getElementBefore(document, offset) + getElementAfter(document, offset);
	}

	/**
	 * Returns identifier at the cursor position (if any).
	 * 
	 * @param document
	 *            document in which element is to be found.
	 * @param offset
	 *            cursor position in the document.
	 * @return identifier at the cursor if any, "" if not found.
	 */
	public static List<String> getElementsAt(IDocument document, int offset) {
		String before = SyntaxUtils.getElementBefore(document, offset);
		int dotPosition = offset - before.length() - 1;
		List<String> elements = SyntaxUtils.getElementsBefore(document, dotPosition);
		String elementAt = SyntaxUtils.getElementAt(document, offset);
		if (elementAt.length() > 0) {
			elements.add(elementAt);
		}

		return elements;
	}

	// /////////////////////////////////////////////////////////////////////////
	// Other
	// ///////////////////////////////////////////////////////////////////////

	/**
	 * Returns the type of the last element in the list, regarding types of it's
	 * predecesors, i.e. it distinguishes <code>Vector.new()</code> from
	 * <code>Dictionary.new()</code> and so on.
	 * 
	 * @param elements
	 *            vector of elements (identifiers).
	 * @return type of the last element in the list.
	 * @throws JDOMException
	 *             if parsing exception occurs (problem with configuration
	 *             file).
	 */
	public static String getLastElementType(List<String> elements) throws JDOMException {

		if (elements.size() == 0) {
			return "";
		}
		Document syntax = Framclipse.getDefault().getFramscriptSyntax();

		int i = elements.size() - 1;
		String type = elements.get(i);

		while ((--i >= 0) && (type != null) && (type.length() > 0)) {
			String nextElement = elements.get(i);
			String query = SyntaxUtils.ROOT;
			query += "/*[@" + SyntaxUtils.NAME_ATTRIBUTE + "='" + type + "']";
			query += "/*[@" + SyntaxUtils.NAME_ATTRIBUTE + "='" + nextElement + "']";
			XPath xpath = XPath.newInstance(query);
			Element element = (Element) xpath.selectSingleNode(syntax);
			type = element.getAttributeValue(SyntaxUtils.TYPE_ATTRIBUTE);
		}
		return type;
	}

	public static String typeName(char type) {
		switch (type) {
		case 'd':
			return "integer";
		case 'f':
			return "float";
		case 's':
			return "string";
		case 'p':
			return "Function";
		case 'o':
			return "object";
		case 'x':
			return "unknown";
		}
		return null;
	}
}
