package com.framsticks.framclipse.editors.codeCompletion;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextViewer;
import org.eclipse.jface.text.TextPresentation;
import org.eclipse.jface.text.contentassist.ICompletionProposal;
import org.eclipse.jface.text.contentassist.IContentAssistProcessor;
import org.eclipse.jface.text.contentassist.IContextInformation;
import org.eclipse.jface.text.contentassist.IContextInformationPresenter;
import org.eclipse.jface.text.contentassist.IContextInformationValidator;
import org.jdom.Attribute;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.xpath.XPath;

import com.framsticks.framclipse.Framclipse;


/**
 * Framscript completion processor.
 */
public class PropertyCompletionProcessor implements IContentAssistProcessor {

	private final Set<String> contexts;
	private final Set<String> objectContexts;
	private final Map<String, String> propertyTypes;

	public PropertyCompletionProcessor(Set<String> contexts, Set<String> objectContexts,
			Map<String, String> propertyTypes) {
		this.contexts = contexts;
		this.objectContexts = objectContexts;
		this.propertyTypes = propertyTypes;
	}

	protected IContextInformationValidator validator = new Validator();

	/*
	 * (non-Javadoc) Method declared on IContentAssistProcessor
	 */
	@SuppressWarnings("unchecked")
	public ICompletionProposal[] computeCompletionProposals(ITextViewer viewer, int documentOffset) {
		// get text before cursor
		IDocument document = viewer.getDocument();
		String elementPart = SyntaxUtils.getElementBefore(document, documentOffset);

		// now if we have parent, ask XPath
		List<OrderableCompletionProposal> proposals = new ArrayList<OrderableCompletionProposal>();

		String query = SyntaxUtils.ROOT + "/" + SyntaxUtils.TYPE_ELEMENT;
		Document syntax = Framclipse.getDefault().getFramscriptSyntax();
		try {
			XPath xpath = XPath.newInstance(query);
			List<Element> list = xpath.selectNodes(syntax);
			for (Element element : list) {
				String name = element.getAttribute(SyntaxUtils.NAME_ATTRIBUTE).getValue();
				if ((name != null) && propertyTypes.keySet().contains(name)) {
					Attribute context = element.getAttribute(SyntaxUtils.CONTEXT_ATTRIBUTE);
					if ((context == null) || contexts.contains(context.getValue())) {
						addTypeCompletions(documentOffset, elementPart, element, proposals);
					}
				}

				String context = element.getAttribute(SyntaxUtils.CONTEXT_ATTRIBUTE).getValue();
				if ((context != null) && objectContexts.contains(context)) {
					addObjectCompletions(documentOffset, elementPart, element, proposals);
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		}

		Collections.sort(proposals);
		ICompletionProposal[] result = new ICompletionProposal[proposals.size()];
		result = proposals.toArray(result);
		return result;
	}

	@SuppressWarnings("unchecked")
	private void addTypeCompletions(int documentOffset, String elementPart, Element typeElement,
			List<OrderableCompletionProposal> proposals) throws JDOMException {

		String typeName = typeElement.getAttribute(SyntaxUtils.NAME_ATTRIBUTE).getValue();
		String objectName = propertyTypes.get(typeName);

		List<Element> fieldsList = typeElement.getChildren(SyntaxUtils.ELEMENT_ELEMENT);
		for (Element element : fieldsList) {
			addCompletion(element, documentOffset, elementPart, objectName, proposals);
		}
	}

	@SuppressWarnings("unchecked")
	private void addObjectCompletions(int documentOffset, String elementPart,
			Element objectElement, List<OrderableCompletionProposal> proposals)
			throws JDOMException {

		String objectName = objectElement.getAttribute(SyntaxUtils.NAME_ATTRIBUTE).getValue();
		addCompletion(objectElement, documentOffset, elementPart, null, proposals);

		List<Element> fieldsList = objectElement.getChildren(SyntaxUtils.ELEMENT_ELEMENT);
		for (Element element : fieldsList) {
			addCompletion(element, documentOffset, elementPart, objectName, proposals);
		}
	}

	@SuppressWarnings("unchecked")
	private void addCompletion(Element element, int documentOffset, String elementPart,
			String objectName, List<OrderableCompletionProposal> proposals) {
		String name = element.getAttributeValue(SyntaxUtils.NAME_ATTRIBUTE);
		String completion = name;
		int cursorPosition = name.length();

		// since we are to propose property completions, we do not want function
		// names here
		String functionValue = element.getAttributeValue(SyntaxUtils.FUNCTION_ATTRIBUTE);
		boolean function = (functionValue != null) && functionValue.equals("true");
		if (function) {
			return;
		}

		String displayString = "";
		if (objectName != null) {
			displayString += objectName + ": ";
		}
		displayString += completion + ":";
		IContextInformation contextInformation = null;

		String type = element.getAttributeValue(SyntaxUtils.TYPE_ATTRIBUTE);
		if ((type != null) && (type.length() > 0)) {
			displayString += "  (" + type + ")";
		}

		if (completion.startsWith(elementPart)) {
			int replacementLength = elementPart.length();
			int replacementOffset = documentOffset - replacementLength;
			OrderableCompletionProposal completionProposal = new OrderableCompletionProposal(
					completion + ": ", replacementOffset, replacementLength, cursorPosition + 2,
					null, displayString, contextInformation, null);

			completionProposal.setType(objectName);
			completionProposal.setName(completion);
			proposals.add(completionProposal);
		}
	}

	/*
	 * (non-Javadoc) Method declared on IContentAssistProcessor
	 */
	public IContextInformation[] computeContextInformation(ITextViewer viewer, int documentOffset) {
		return null;
	}

	/*
	 * (non-Javadoc) Method declared on IContentAssistProcessor
	 */
	public char[] getCompletionProposalAutoActivationCharacters() {
		return new char[] { '.' };
	}

	/*
	 * (non-Javadoc) Method declared on IContentAssistProcessor
	 */
	public char[] getContextInformationAutoActivationCharacters() {
		return new char[] {};
	}

	/*
	 * (non-Javadoc) Method declared on IContentAssistProcessor
	 */
	public IContextInformationValidator getContextInformationValidator() {
		return validator;
	}

	/*
	 * (non-Javadoc) Method declared on IContentAssistProcessor
	 */
	public String getErrorMessage() {
		return null;
	}

	/**
	 * Simple content assist tip closer. The tip is valid in a range of 5
	 * characters around its popup location.
	 */
	protected static class Validator implements IContextInformationValidator,
			IContextInformationPresenter {

		protected int installOffset;

		/*
		 * @see IContextInformationValidator#isContextInformationValid(int)
		 */
		public boolean isContextInformationValid(int offset) {
			return Math.abs(installOffset - offset) < 5;
		}

		/*
		 * @see IContextInformationValidator#install(IContextInformation,
		 *      ITextViewer, int)
		 */
		public void install(IContextInformation info, ITextViewer viewer, int offset) {
			installOffset = offset;
		}

		/*
		 * @see org.eclipse.jface.text.contentassist.IContextInformationPresenter#updatePresentation(int,
		 *      TextPresentation)
		 */
		public boolean updatePresentation(int documentPosition, TextPresentation presentation) {
			return false;
		}
	}
}
