package com.framsticks.structure;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.framsticks.params.Access;
import com.framsticks.params.CompositeParam;
import com.framsticks.params.EventListener;
import com.framsticks.params.FramsClass;
import com.framsticks.params.ParamBuilder;
import com.framsticks.params.PrimitiveParam;
import com.framsticks.params.annotations.AutoAppendAnnotation;
import com.framsticks.params.annotations.FramsClassAnnotation;
import com.framsticks.params.types.EventParam;
import com.framsticks.params.types.ProcedureParam;
import com.framsticks.util.FramsticksException;
import com.framsticks.util.dispatching.Dispatcher;
import com.framsticks.util.dispatching.DispatcherSetable;
import com.framsticks.util.dispatching.Dispatching;
import com.framsticks.util.dispatching.FutureHandler;
import com.framsticks.util.dispatching.Joinable;
import com.framsticks.util.dispatching.JoinableState;
import com.framsticks.util.lang.Casting;

import static com.framsticks.structure.TreeOperations.*;

@FramsClassAnnotation
public final class LocalTree extends AbstractTree {
	private static final Logger log = LogManager.getLogger(LocalTree.class);

	protected Joinable joinableRootObject;

	/**
	 *
	 */
	public LocalTree() {
		super();
		bufferedDispatcher.setBuffer(false);
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	@AutoAppendAnnotation
	public void setRootObject(Object object) {
		final Class<?> javaClass = object.getClass();
		registry.registerAndBuild(javaClass);

		Access access = registry.createAccess(javaClass);

		assignRootParam(access.buildParam(new ParamBuilder()).id(getName()).finish(CompositeParam.class));
		assignRootObject(object);

		if (object instanceof Joinable) {
			joinableRootObject = (Joinable) object;
		}
		if (object instanceof DispatcherSetable) {
			DispatcherSetable<?> setable = (DispatcherSetable<?>) object;
			setable.setDispatcher((Dispatcher) this);
		}
	}

	public Object getRootObject() {
		Object result = getAssignedRoot().getObject();
		if (result == null) {
			throw new FramsticksException().msg("object tree is empty").arg("tree", this);
		}
		return result;
	}

	public <T> T getRootObject(Class<T> type) {
		Object result = getRootObject();
		if (!type.isInstance(result)) {
			throw new FramsticksException().msg("object tree holds object of different kind").arg("object", result).arg("requested", type).arg("tree", this);
		}
		return type.cast(result);
	}

	@Override
	public void get(Path path, FutureHandler<Path> future) {
		assert isActive();
		log.debug("requesting: {}", path);
		path = resolveTopSync(path);
		future.pass(path);
	}

	// @Override
	// public void get(Path path, ValueParam param, Future<Object> future) {
	//	assert isActive();
	//	path = resolveTopSync(path);
	//	future.pass(bindAccess(path).get(param, Object.class));
	// }

	@Override
	public <R> void call(Path path, ProcedureParam param, Object[] arguments, Class<R> resultType, FutureHandler<R> future) {
		assert isActive();
		try {
			Object result = bindAccess(path).call(param, arguments);
			future.pass(Casting.nullOrThrowCast(resultType, result));
		} catch (FramsticksException e) {
			future.handle(e);
		}
	}

	@Override
	public void info(Path path, FutureHandler<FramsClass> future) {
		assert isActive();
		Path p = path.tryResolveIfNeeded();
		Class<?> javaClass = p.getTopObject().getClass();
		FramsClass framsClass = registry.registerReflectedIfNeeded(javaClass);
		if (framsClass != null) {
			future.pass(framsClass);
		} else {
			future.handle(new FramsticksException().msg("failed to find info for class").arg("java class", javaClass));
		}
	}

	protected Path resolveTopSync(Path path) {
		assert isActive();
		assert path.isOwner(this);
		if (path.getTop().getObject() != null) {
			return path;
		}
		Access access = bindAccess(path.getUnder());
		Object object = access.get(path.getTop().getParam(), Object.class);
		if (object == null) {
			throw new FramsticksException().msg("failed to resolve").arg("path", path);
		}
		return path.appendResolution(object);
	}


	@Override
	public void set(Path path, PrimitiveParam<?> param, Object value, final FutureHandler<Integer> future) {
		assert isActive();
		future.pass(bindAccess(path).set(param, value));
	}

	public <A> void addListener(Path path, EventParam param, EventListener<A> listener, Class<A> argumentType, FutureHandler<Void> future) {
		assert isActive();
		try {
			bindAccess(path).reg(param, listener);
			future.pass(null);
		} catch (FramsticksException e) {
			future.handle(e);
		}
	}

	public void removeListener(Path path, EventParam param, EventListener<?> listener, FutureHandler<Void> future) {
		assert isActive();
		try {
			bindAccess(path).regRemove(param, listener);
			future.pass(null);
		} catch (FramsticksException e) {
			future.handle(e);
		}
	}

	@Override
	protected void joinableStart() {
		super.joinableStart();
		if (joinableRootObject != null) {
			Dispatching.use(joinableRootObject, this);
		}
	}

	@Override
	protected void joinableInterrupt() {
		if (joinableRootObject != null) {
			Dispatching.drop(joinableRootObject, this);
		}
		super.joinableInterrupt();
	}

	@Override
	protected void joinableFinish() {
		super.joinableFinish();
	}

	@Override
	protected void joinableJoin() throws InterruptedException {
		if (joinableRootObject != null) {
			Dispatching.join(joinableRootObject);
		}
		super.joinableJoin();
	}

	@Override
	public void childChangedState(Joinable joinable, JoinableState state) {
		super.childChangedState(joinable, state);
		if (joinable == joinableRootObject) {
			proceedToState(state);
		}
	}

}
