package com.framsticks.params;

import com.framsticks.util.FramsticksException;
import com.framsticks.util.Misc;
import com.framsticks.util.lang.Numbers;
import com.framsticks.util.lang.Strings;

import javax.annotation.concurrent.Immutable;

/**
 * @author Piotr Sniegowski
 */
@Immutable
public abstract class PrimitiveParam<S> extends ValueParam {

	/** The minimum allowed value of parameter. */
	protected final Object min;

	/** The maximum allowed value of parameter. */
	protected final Object max;

	/** The default value of parameter. */
	protected final Object def;

	/**
	 * @param builder
	 */
	public PrimitiveParam(ParamBuilder builder) {
		super(builder);
		Class<?> storageType = Misc.returnNotNull(builder.getStorageType(), Object.class);
		min = tryCastAndReturn(builder.getMin(), storageType);
		max = tryCastAndReturn(builder.getMax(), storageType);
		def = tryCastAndReturn(builder.getDef(), storageType);
	}

	// public void save(Sink sink, Object value) {
	// 	sink.print(value);
	// }

	protected <T> T tryCastAndReturn(Object value, Class<T> type) {
		if (value == null) {
			return null;
		}
		try {
			if (type.equals(Object.class)) {
				return type.cast(value);
			}
			if (value instanceof String) {
				if (type.equals(String.class)) {
					return type.cast(value);
				}
				return Numbers.parse((String) value, type);
			} if ((value instanceof Integer) && (type.equals(Double.class))) {
				return type.cast(new Double((Integer) value));
			} else {
				return type.cast(value);
			}
		} catch (ClassCastException e) {
			throw new FramsticksException().msg("failed to cast").cause(e).arg("param", this).arg("actual", value.getClass()).arg("requested", type).arg("value", value);
		}
	}

	/**
	 * Gets the minimum value of parameter.
	 *
	 * @param <T> the generic getType which must be correctly specified by user
	 * @param type the getType of variable, can be checked by
	 * @return the minimum allowed value of parameter
	 * @throws ClassCastException the class cast exception, thrown when given getType is incorrect
	 */
	public <T> T getMin(Class<T> type) {
		return tryCastAndReturn(min, type);
	}

	/**
	 * Gets the maximum value of parameter.
	 *
	 * @param <T> the generic getType which must be correctly specified by user
	 * @param type the getType of variable, can be checked by {@link #getStorageType()}
	 * @return the maximum allowed value of parameter
	 * @throws ClassCastException the class cast exception, thrown when given getType is incorrect
	 */
	public <T> T getMax(Class<T> type) {
		return tryCastAndReturn(max, type);
	}

	/**
	 * Gets the default value of parameter.
	 *
	 * @param <T> the generic getType which must be correctly specified by user
	 * @param type the getType of variable, can be checked by {@link #getStorageType()}
	 * @return the default value of parameter
	 * @throws ClassCastException the class cast exception, thrown when given getType is incorrect
	 */
	public <T> T getDef(Class<T> type) {
		return tryCastAndReturn(def, type);
	}

	public <T> String serialize(T value) {
		return Strings.toStringNullProof(value);
	}

}
