package com.framsticks.gui.console;

import java.awt.BorderLayout;
import java.awt.Dimension;

import javax.swing.BorderFactory;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextPane;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.framsticks.communication.Connection;
import com.framsticks.gui.FrameJoinable;
import com.framsticks.params.annotations.FramsClassAnnotation;
import com.framsticks.util.ExceptionHandler;
import com.framsticks.util.FramsticksException;
import com.framsticks.util.dispatching.Dispatching;
import com.framsticks.util.dispatching.Joinable;
import com.framsticks.util.dispatching.JoinableParent;
import com.framsticks.util.dispatching.JoinableState;
import com.framsticks.util.dispatching.RunAt;

@FramsClassAnnotation
public abstract class Console extends FrameJoinable implements JoinableParent {
	private static final Logger log = LogManager.getLogger(Console.class);

	/**
	 * Painter coloring manager responses before display.
	 */
	protected ConsolePainter consolePainter;

	protected Connection connection;

	protected JPanel panel;

	/**
	 * @param connection
	 */
	public Console() {
		setTitle("console");
	}

	/**
	 * @return the connection
	 */
	public Connection getConnection() {
		return connection;
	}

	@Override
	public String getName() {
		return connection != null ? "console for " + connection.getName() : "console";
	}

	@Override
	protected void initializeGui() {
		super.initializeGui();
		panel = new JPanel();
		panel.setLayout(new BorderLayout());
		panel.setSize(new Dimension(440, 400));
		panel.setMinimumSize(new Dimension(440, 400));

		JTextPane text = new JTextPane();
		consolePainter = new ConsolePainter(text);

		text.setEditable(false);
		final JScrollPane scrollText = new JScrollPane(text);
		scrollText.setBorder(BorderFactory.createEtchedBorder());

		JPanel scrollPanel = new JPanel();
		scrollPanel.setLayout(new BorderLayout());
		scrollPanel.add(scrollText, BorderLayout.CENTER);
		scrollPanel.setBorder(BorderFactory.createEmptyBorder(7, 7, 7, 7));

		panel.add(scrollPanel, BorderLayout.CENTER);

		getSwing().getContentPane().add(panel, BorderLayout.CENTER);

		log.debug("initialized gui");
	}

	@Override
	public void childChangedState(Joinable joinable, JoinableState state) {
		if (joinable == connection) {
			proceedToState(state);
		}
	}

	protected ExceptionHandler getExceptionHandler() {
		return new ExceptionHandler() {

			@Override
			public void handle(FramsticksException exception) {
				throw exception;

			}
		};
	}

	@Override
	protected void joinableStart() {
		if (connection == null) {
			throw new FramsticksException().msg("connection was not set").arg("console", this);
		}
		super.joinableStart();
		Dispatching.use(connection, this);
	}

	@Override
	protected void joinableInterrupt() {
		Dispatching.drop(connection, this);
		super.joinableInterrupt();
	}

	@Override
	protected void joinableFinish() {
		super.joinableFinish();
	}

	@Override
	protected void joinableJoin() throws InterruptedException {
		Dispatching.join(connection);
		super.joinableJoin();
	}

	protected void dispatchWrite(final String line) {
		dispatch(new RunAt<Console>(getExceptionHandler()) {
			@Override
			protected void runAt() {
				consolePainter.userLine(line);
			}
		});
	}

}
