package com.framsticks.gui;

import java.awt.Dimension;
import java.util.Collection;
import java.util.Map;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.border.TitledBorder;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.framsticks.gui.controls.CheckBoxControl;
import com.framsticks.gui.controls.Control;
import com.framsticks.gui.controls.ControlOwner;
import com.framsticks.gui.controls.EnumControl;
import com.framsticks.gui.controls.EventControl;
import com.framsticks.gui.controls.ProcedureControl;
import com.framsticks.gui.controls.SliderControl;
import com.framsticks.gui.controls.TextAreaControl;
import com.framsticks.gui.controls.TextFieldControl;
import com.framsticks.params.CompositeParam;
import com.framsticks.params.Param;
import com.framsticks.params.ParamFlags;
import com.framsticks.params.PrimitiveParam;
import com.framsticks.params.types.BinaryParam;
import com.framsticks.params.types.BooleanParam;
import com.framsticks.params.types.ColorParam;
import com.framsticks.params.types.DecimalParam;
import com.framsticks.params.types.EnumParam;
import com.framsticks.params.types.EventParam;
import com.framsticks.params.types.FloatParam;
import com.framsticks.params.types.ProcedureParam;
import com.framsticks.params.types.StringParam;
import com.framsticks.params.types.UniversalParam;
import com.framsticks.util.FramsticksException;
import com.framsticks.util.lang.Strings;

public final class Gui {

	private static final Logger log = LogManager.getLogger(Gui.class.getName());

	private Gui() {
	}

	public static Control createComponentForText(PrimitiveParam<?> valueParam) {
		if (valueParam.getMax(Object.class) != null) {
			return new TextFieldControl(valueParam);
		}
		return new TextAreaControl(valueParam);


		// if (valueParam.getMin(Object.class) != null) {
		// 	return new TextAreaControl(valueParam);
		// }
		// return new TextFieldControl(valueParam);
	}

	public static Control createSuitable(Param param) {

		if (param instanceof EnumParam) {
			return new EnumControl((EnumParam) param);
		}
		if (param instanceof BooleanParam) {
			return new CheckBoxControl((BooleanParam) param);
		}
		if (param instanceof DecimalParam) {
			DecimalParam decimalParam = (DecimalParam)param;
			if (decimalParam.getMin(Integer.class) != null && decimalParam.getMax(Integer.class) != null) {
				return new SliderControl(decimalParam);
			}
			return new TextFieldControl(decimalParam);
		}
		if (param instanceof FloatParam) {
			FloatParam floatParam = (FloatParam)param;
			if (floatParam.getMin(Double.class) != null && floatParam.getMax(Double.class) != null) {
				return new SliderControl(floatParam);
			}
			return new TextFieldControl(floatParam);
		}
		if (param instanceof StringParam) {
			return createComponentForText((StringParam) param);
		}
		if (param instanceof ProcedureParam) {
			return new ProcedureControl((ProcedureParam) param);
		}
		if (param instanceof BinaryParam) {
			return new TextFieldControl((BinaryParam) param);
		}
		if (param instanceof ColorParam) {
			return new TextFieldControl((ColorParam) param);
		}
		if (param instanceof UniversalParam) {
			return new TextAreaControl((UniversalParam) param);
		}
		if (param instanceof EventParam) {
			return new EventControl((EventParam) param);
		}
		return null;
	}

	public static <P extends Param, C extends Control> void fillWithControls(ControlOwner owner, JPanel panel, Collection<P> params, Map<String, C> components, Class<C> controlType) {
		panel.setLayout(new BoxLayout(panel, BoxLayout.PAGE_AXIS));
		for (P param : params) {
			if (param.hasFlag(ParamFlags.USERHIDDEN)) {
				continue;
			}
			assert !(param instanceof CompositeParam);
			Control c = Gui.createSuitable(param);

			if (!controlType.isInstance(c)) {
				throw new FramsticksException().msg("created control is not of required type").arg("control", c).arg("type", controlType);
			}

			C control = controlType.cast(c);

			control.setOwner(owner);

			log.debug("add component for {}", param);
			JPanel line = new JPanel();
			line.setLayout(new BoxLayout(line, BoxLayout.LINE_AXIS));
			line.setAlignmentX(JPanel.LEFT_ALIGNMENT);

			line.add(control);
			line.revalidate();
			panel.add(line);
			panel.add(Box.createRigidArea(new Dimension(0, 8)));
			components.put(param.getId(), control);
			//component.setAlignmentX(LEFT_ALIGNMENT);
			// components.put(param.getId(), control);
		}

	}

	public static String getBestName(Param param) {
		if (Strings.notEmpty(param.getName())) {
			return param.getName();
		}
		if (Strings.notEmpty(param.getId())) {
			return param.getId();
		}
		return "?";
	}

	public static void setupTitledControl(Control control, JComponent... components) {

		control.setLayout(new BoxLayout(control, BoxLayout.PAGE_AXIS));
		control.setBorder(new TitledBorder(Gui.getBestName(control.getParam())));
		for (JComponent c : components) {
			// control.add(Box.createRigidArea(new Dimension(0, 4)));
			control.add(c);
		}
	}

	public static void layoutInRow(JPanel panel, JComponent first, JComponent... components) {
		panel.setLayout(new BoxLayout(panel, BoxLayout.LINE_AXIS));
		panel.setAlignmentX(JPanel.LEFT_ALIGNMENT);

		panel.add(first);
		for (JComponent c : components) {
			panel.add(Box.createRigidArea(new Dimension(8, 0)));
			panel.add(c);
		}
	}

	public static void addLeftToLabel(Control control, JComponent... components) {

		JLabel label = new JLabel(getBestName(control.getParam()));
		label.setToolTipText(control.getToolTipText());
		label.setHorizontalAlignment(JLabel.LEFT);
		Dimension labelSize = new Dimension(150, 30);
		label.setMaximumSize(labelSize);
		label.setMinimumSize(labelSize);
		label.setPreferredSize(labelSize);

		layoutInRow(control, label, components);

	}
}
