package com.framsticks.gui;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.swing.JFrame;
import javax.swing.UIManager;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.framsticks.util.ExceptionHandler;
import com.framsticks.util.FramsticksException;
import com.framsticks.util.dispatching.RunAt;
import com.framsticks.util.dispatching.ThrowExceptionHandler;

public abstract class FrameJoinable extends SwingJoinable<JFrame> implements ExceptionHandler {
	private static final Logger log =
		LogManager.getLogger(FrameJoinable.class);

	protected String title;

	protected final StatusBar statusBar;


	/**
	 *
	 */
	public FrameJoinable() {
		statusBar = new StatusBar(this);
	}

	public void setTitle(String title) {
		this.title = title;
		if (hasSwing()) {
			getSwing().setTitle(title);
		}
	}

	/**
	 * @return the statusBar
	 */
	public StatusBar getStatusBar() {
		return statusBar;
	}

	@Override
	public String getName() {
		return title;
	}

	private final static AtomicBoolean sentGuiInitialization = new AtomicBoolean(false);


	@Override
	protected void joinableStart() {

		if (sentGuiInitialization.compareAndSet(false, true)) {
			dispatch(new RunAt<FrameJoinable>(ThrowExceptionHandler.getInstance()) {
				@Override
				protected void runAt() {
					try {
						boolean found = false;
						// for (LookAndFeelInfo info : UIManager.getInstalledLookAndFeels()) {
						//	log.info("look and feel available: {}", info.getName());
						//	if ("Nimbus".equals(info.getName())) {
						//		UIManager.setLookAndFeel(info.getClassName());
						//		found = true;
						//		break;
						//	}
						// }
						if (!found) {
							UIManager.setLookAndFeel("com.sun.java.swing.plaf.gtk.GTKLookAndFeel");
						}
						JFrame.setDefaultLookAndFeelDecorated(true);
					} catch (Exception e) {
						throw new FramsticksException().msg("failed to initialize Look&Feel").cause(e);
					}
				}
			});
		}

		super.joinableStart();
		dispatch(new RunAt<FrameJoinable>(ThrowExceptionHandler.getInstance()) {
			@Override
			protected void runAt() {
				getSwing().setVisible(true);
			}
		});

	}

	@Override
	protected void joinableInterrupt() {
		super.joinableInterrupt();

		dispatch(new RunAt<Frame>(ThrowExceptionHandler.getInstance()) {
			@Override
			protected void runAt() {
				finishJoinable();
			}
		});

	}

	@Override
	protected void joinableFinish() {
		super.joinableFinish();
		log.debug("disposing frame {}", this);
		getSwing().dispose();
	}

	@Override
	protected void joinableJoin() throws InterruptedException {
		super.joinableJoin();

	}

	@Override
	protected void initializeGui() {
		setSwing(new JFrame(title));
		statusBar.initializeGui();

		Container contentPane = getSwing().getContentPane();
		contentPane.setLayout(new BorderLayout());
		contentPane.add(statusBar.getSwing(), BorderLayout.SOUTH);

		getSwing().setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);

		getSwing().addWindowListener(new WindowAdapter() {
			@Override
			public void windowClosing(WindowEvent e) {
				log.info("received closing");
				interruptJoinable();
			}
		});


	}

	@Override
	public void handle(FramsticksException exception) {
		statusBar.handle(exception);
	}

}
