package com.framsticks.experiment;

import com.framsticks.communication.File;
import com.framsticks.communication.queries.NeedFile;
import com.framsticks.communication.queries.NeedFileAcceptor;
import com.framsticks.params.AccessOperations;
import com.framsticks.params.CastFailure;
import com.framsticks.params.EventListener;
import com.framsticks.params.FramsClass;
import com.framsticks.params.UniqueObject;
import com.framsticks.params.annotations.FramsClassAnnotation;
import com.framsticks.params.annotations.ParamAnnotation;
import com.framsticks.params.types.BooleanParam;
import com.framsticks.params.types.EventParam;
import com.framsticks.params.types.ProcedureParam;
import com.framsticks.remote.RemoteTree;
import com.framsticks.structure.Path;
import com.framsticks.structure.Tree;
import com.framsticks.structure.messages.ListChange;
import com.framsticks.structure.messages.ValueChange;
import com.framsticks.util.ExceptionHandler;
import com.framsticks.util.FramsticksException;
import com.framsticks.util.dispatching.AbstractJoinable;
import com.framsticks.util.dispatching.Dispatcher;
import com.framsticks.util.dispatching.Dispatching;
import com.framsticks.util.dispatching.FutureHandler;
import com.framsticks.util.dispatching.Future;
import com.framsticks.util.dispatching.Joinable;
import com.framsticks.util.dispatching.JoinableParent;
import com.framsticks.util.dispatching.JoinableState;
import com.framsticks.util.dispatching.RunAt;
import com.framsticks.util.dispatching.ThrowExceptionHandler;
import com.framsticks.util.lang.Holder;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import static com.framsticks.params.ParamsUtil.arguments;
import static com.framsticks.params.ParamsUtil.getParam;
import static com.framsticks.structure.TreeOperations.*;

@FramsClassAnnotation
public final class Simulator extends AbstractJoinable implements Dispatcher<Simulator>, JoinableParent, UniqueObject, ExceptionHandler {

	private static final Logger log = LogManager.getLogger(Simulator.class);

	protected String uid;

	protected final RemoteTree remoteTree;
	protected final Path simulatorPath;
	protected final FramsClass simulatorClass;
	protected final Experiment experiment;
	protected final EventListener<ValueChange> runningListener;

	/**
	 *
	 */
	public Simulator(Experiment experiment, RemoteTree remoteTree, Path simulatorPath) {
		super();
		this.remoteTree = remoteTree;
		this.simulatorPath = simulatorPath.assureResolved();
		this.experiment = experiment;
		this.simulatorClass = getFramsClass(simulatorPath);

		assert remoteTree.isActive();
		assert experiment.isActive();

		log.info("simulator ready {}", this);

		runningListener = new EventListener<ValueChange>() {
			@Override
			public void action(ValueChange argument) {
				try {
					boolean running = getParam(simulatorClass, "running", BooleanParam.class).reassign(argument.value, null).getValue();
					log.debug("running state of {} changed: {}", Simulator.this, running);
					if (!running) {
						Simulator.this.experiment.simulators.fireChildrenChange(Simulator.this, ListChange.Action.Modify, "ready", "stoped");
					}
				} catch (CastFailure e) {
					log.error("failure: ", e);
				}
			}
		};

		addListener(simulatorPath, getParam(simulatorClass, "running_changed", EventParam.class), runningListener, ValueChange.class, new Future<Void>(this) {
			@Override
			protected void result(Void result) {
				log.debug("running listener for {} registered", this);
			}
		});
	}

	@ParamAnnotation
	public String getAddress() {
		return remoteTree.getAddress();
	}

	@Override
	@ParamAnnotation
	public String getName() {
		return getAddress();
	}

	@Override
	@ParamAnnotation
	public String getUid() {
		return uid;
	}

	@Override
	public void setUid(String uid) {
		this.uid = uid;
	}

	/**
	 * @return the tree
	 */
	@ParamAnnotation
	public RemoteTree getRemoteTree() {
		return remoteTree;
	}

	/**
	 * @return the simulatorPath
	 */
	public Path getSimulatorPath() {
		return simulatorPath;
	}

	/**
	 * @return the simulatorClass
	 */
	public FramsClass getSimulatorClass() {
		return simulatorClass;
	}

	@Override
	protected void joinableStart() {
		Dispatching.use(remoteTree, this);
	}

	@Override
	protected void joinableInterrupt() {
		Dispatching.drop(remoteTree, this);

	}

	@Override
	protected void joinableFinish() {

	}

	@Override
	protected void joinableJoin() throws InterruptedException {
		Dispatching.join(remoteTree);
	}

	@ParamAnnotation(paramType = ProcedureParam.class)
	public void init() {
		log.debug("initializing simulator {}", this);
		call(simulatorPath, "init", arguments(), Object.class, Future.doNothing(Object.class, this));
	}

	@ParamAnnotation(paramType = ProcedureParam.class)
	public void start() {
		log.debug("starting simulator {}", this);
		call(simulatorPath, "start", arguments(), Object.class, Future.doNothing(Object.class, this));
	}

	@ParamAnnotation(paramType = ProcedureParam.class)
	public void stop() {
		log.debug("stoping simulator {}", this);
		call(simulatorPath, "stop", arguments(), Object.class, Future.doNothing(Object.class, this));
	}

	@ParamAnnotation(paramType = ProcedureParam.class)
	public void abort() {
		assert isActive();
		log.info("explicitly aborting {}", this);
		experiment.removeSimulator(this);
		interruptJoinable();
	}

	@Override
	public void childChangedState(Joinable joinable, JoinableState state) {
		proceedToState(state);
	}

	@Override
	public void handle(FramsticksException exception) {
		experiment.handle(new FramsticksException().msg("exception caught in simulator").arg("simulator", this).cause(exception));
	}

	@Override
	public boolean isActive() {
		return experiment.isActive();
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	@Override
	public void dispatch(RunAt<? extends Simulator> runnable) {
		experiment.dispatch((RunAt) runnable);
	}

	protected final AtomicInteger netloadIdCounter = new AtomicInteger();

	public <N> void netload(final N net, final FutureHandler<Object> future) {
		final String netloadId = "netload" + netloadIdCounter.getAndIncrement();

		final File file = AccessOperations.convert(File.class, net, getRemoteTree().getRegistry());
		log.debug("uploading file {} to {} identified by {}", file, simulatorPath, netloadId);

		final Holder<NeedFileAcceptor> acceptor = new Holder<>();
		final Tree tree = simulatorPath.getTree();

		acceptor.set(new NeedFileAcceptor() {

			@Override
			public boolean acceptNeed(NeedFile needFile) {
				if (!needFile.getDescription().equals(netloadId)) {
					return false;
				}
				log.debug("accepting netload {}", netloadId);
				needFile.getFuture().pass(file);
				tree.dispatch(new RunAt<Tree>(ThrowExceptionHandler.getInstance()) {

					@Override
					protected void runAt() {
						tree.removeNeedFileAcceptor(acceptor.get());
					}
				});
				return true;
			}

		});

		simulatorPath.getTree().addNeedFileAcceptor(Integer.MIN_VALUE, acceptor.get());

		call(simulatorPath, getParam(simulatorPath, "netload_id", ProcedureParam.class), arguments(netloadId), Object.class, new Future<Object>(future) {

			@Override
			protected void result(Object result) {
				log.debug("netload of {} done", file);
				future.pass(result);
			}
		});

	}

	public <N> void netsave(Class<N> netJavaClass, final FutureHandler<N> futureNet) {
		call(simulatorPath, getParam(simulatorPath, "netsave", ProcedureParam.class), arguments(), netJavaClass, new Future<N>(futureNet) {

			@Override
			protected void result(N net) {
				log.debug("netsave of {} done", net);
				futureNet.pass(net);
			}
		});
	}
}
