package com.framsticks.framclipse.editors.codeCompletion;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextViewer;
import org.eclipse.jface.text.TextPresentation;
import org.eclipse.jface.text.contentassist.ContextInformation;
import org.eclipse.jface.text.contentassist.ICompletionProposal;
import org.eclipse.jface.text.contentassist.IContentAssistProcessor;
import org.eclipse.jface.text.contentassist.IContextInformation;
import org.eclipse.jface.text.contentassist.IContextInformationPresenter;
import org.eclipse.jface.text.contentassist.IContextInformationValidator;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IFileEditorMapping;
import org.eclipse.ui.IPathEditorInput;
import org.jdom.Attribute;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.xpath.XPath;

import com.framsticks.framclipse.Framclipse;
import com.framsticks.framclipse.editors.EditorType;
import com.framsticks.framclipse.editors.FramclipseEditor;
import com.framsticks.framclipse.internal.parser.ASTFObject;
import com.framsticks.framclipse.internal.parser.ASTGlobalInclude;
import com.framsticks.framclipse.internal.parser.ASTProperty;
import com.framsticks.framclipse.internal.parser.FramclipseParser;
import com.framsticks.framclipse.internal.parser.Node;
import com.framsticks.framclipse.syntaxColoring.FramscriptCodeScanner;


/**
 * Framscript completion processor.
 */
public class FramscriptCompletionProcessor implements IContentAssistProcessor {

	private final Set<String> contexts;

	protected final FramclipseEditor editor;

	protected final EditorType editorType;

	protected FramclipseParser parser;

	public FramscriptCompletionProcessor(Set<String> contexts,
			FramclipseEditor editor, EditorType editorType) {
		super();
		this.contexts = contexts;
		this.editor = editor;
		this.editorType = editorType;
	}

	protected IContextInformationValidator validator = new Validator();

	/*
	 * (non-Javadoc) Method declared on IContentAssistProcessor
	 */
	@SuppressWarnings("unchecked")
	public ICompletionProposal[] computeCompletionProposals(ITextViewer viewer,
			int documentOffset) {

		// get text before cursor
		IDocument document = viewer.getDocument();
		String elementPart = SyntaxUtils.getElementBefore(document,
				documentOffset);

		// check if there is a dot before this text -> then fetch parent
		int offset = documentOffset - elementPart.length() - 1;

		String parent = null;
		List<String> elementsBefore = SyntaxUtils.getElementsBefore(document,
				offset);
		try {
			parent = SyntaxUtils.getLastElementType(elementsBefore);
		} catch (Exception e1) {
			e1.printStackTrace();
		}

		// now if we have parent, ask XPath
		List<OrderableCompletionProposal> proposals = new ArrayList<OrderableCompletionProposal>();
		if (EditorType.EXPDEF.equals(editorType) && "ExpParams".equals(parent)) {
			addExpParamsProposals(viewer.getDocument(), documentOffset,
					elementPart, proposals);
		} else {
			String query = SyntaxUtils.ROOT + "/*";
			if ((parent != null) && (parent.length() > 0)) {
				query += "[@" + SyntaxUtils.NAME_ATTRIBUTE + "='" + parent
						+ "']/" + SyntaxUtils.ELEMENT_ELEMENT;
			}
			Document syntax = Framclipse.getDefault().getFramscriptSyntax();
			try {
				XPath xpath = XPath.newInstance(query);
				List<Element> list = xpath.selectNodes(syntax);
				for (Element element : list) {
					Attribute context = element
							.getAttribute(SyntaxUtils.CONTEXT_ATTRIBUTE);
					if ((context == null)
							|| contexts.contains(context.getValue())) {
						addCompletion(element, documentOffset, elementPart,
								proposals);
					}
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		if (parent == null || parent.length() == 0)
			addKeywordProposals(documentOffset, elementPart, proposals);

		Collections.sort(proposals);

		ICompletionProposal[] result = new ICompletionProposal[proposals.size()];
		result = proposals.toArray(result);
		return result;
	}

	void addExpParamsProposals(IDocument document, int documentOffset,
			String elementPart, List<OrderableCompletionProposal> proposals) {
		Node model = editor.getFileModel();
		if (model == null) 
			return;
		processExpParamNode(model, documentOffset, elementPart, proposals);
	}

	static void processExpParamNode(Node model, int documentOffset,
			String elementPart, List<OrderableCompletionProposal> proposals) {
		{
			final int modelNumChild = model.jjtGetNumChildren();
			for (int i = 0; i < modelNumChild; ++i) {
				final Node child = model.jjtGetChild(i);
				if (child instanceof ASTFObject) {
					final ASTFObject fo = (ASTFObject) child;
					if ("prop".equals(fo.getClassName())) {
						String id = "", name = null, type = null;
						final int numChildren = fo.jjtGetNumChildren();
						for (int j = 0; j < numChildren; j++) {
							final Node fochild = fo.jjtGetChild(j);
							if (fochild instanceof ASTProperty) {
								final ASTProperty prop = (ASTProperty) fochild;
								final String pname = prop.getName();
								if ("id".equalsIgnoreCase(pname)) {
									id = prop.getValue();
									if (id == null || id.length() == 0
											|| !id.startsWith(elementPart)) {
										break;
									}
								} else if ("name".equalsIgnoreCase(pname)) {
									name = prop.getValue();
								} else if ("type".equalsIgnoreCase(pname)) {
									type = prop.getValue();
								}
							}
						}
						addExpdefCompletion(documentOffset, elementPart,
								proposals, id, name, type);
					}
				} else if (child instanceof ASTGlobalInclude) {
					processExpParamNode(child, documentOffset, elementPart,
							proposals);
				}
			}
		}
	}

	static void addExpdefCompletion(int documentOffset, String elementPart,
			List<OrderableCompletionProposal> proposals, String id,
			String name, String type) {

		if (id == null || id.length() == 0 || !id.startsWith(elementPart))
			return;
		String descr = id;
		if (type != null) {
			type = type.trim();
			final char t = type.charAt(0);
			final String typeName = SyntaxUtils.typeName(t);
			if (typeName != null)
				type = typeName;
			descr += " " + type;
		}
		 if (name != null)
		 descr += " (" + name + ")";
		final OrderableCompletionProposal prop = new OrderableCompletionProposal(
				id, documentOffset - elementPart.length(),
				elementPart.length(), id.length() - 1, null, descr, null, null);
		prop.setName(id);
		proposals.add(prop);
	}

	void addKeywordProposals(int documentOffset, String elementPart,
			List<OrderableCompletionProposal> proposals) {
		for (String kwd : FramscriptCodeScanner.keywords) {
			if (kwd.startsWith(elementPart)) {
				OrderableCompletionProposal prop = getProposalForKeyword(kwd,
						documentOffset, elementPart);
				proposals.add(prop);
			}
		}
	}

	/**
	 * Function to customize the creation of completion proposals based on the
	 * current keyword. For example, the <code>if</code> keyword gets some
	 * additional parentheses.
	 * 
	 * @param kwd
	 * @param documentOffset
	 * @param elementPart
	 * @return
	 */
	OrderableCompletionProposal getProposalForKeyword(String kwd,
			int documentOffset, String elementPart) {
		OrderableCompletionProposal prop = null;

		if ("if".equals(kwd)) {
			String repString = kwd + " ()";
			prop = new OrderableCompletionProposal(repString, documentOffset
					- elementPart.length(), elementPart.length(), repString
					.length() - 1, null, kwd + " keyword", null, null);
		} else {
			prop = new OrderableCompletionProposal(kwd, documentOffset
					- elementPart.length(), elementPart.length(), kwd.length(),
					null, kwd + " keyword", null, null);
		}

		prop.setName(kwd);
		return prop;
	}

	@SuppressWarnings("unchecked")
	private void addCompletion(Element element, int documentOffset,
			String elementPart, List<OrderableCompletionProposal> proposals) {
		String name = element.getAttributeValue(SyntaxUtils.NAME_ATTRIBUTE);
		String completion = name;
		int cursorPosition = name.length();
		String functionValue = element
				.getAttributeValue(SyntaxUtils.FUNCTION_ATTRIBUTE);
		boolean function = (functionValue != null)
				&& functionValue.equals("true");
		String displayString = "";
		IContextInformation contextInformation = null;
		displayString = completion;
		if (function) {
			completion += "()";
			displayString += "(";
			Element arguments = element.getChild(SyntaxUtils.ARGUMENTS_ELEMENT);
			if (arguments != null) {
				List<Element> argumentsList = arguments
						.getChildren(SyntaxUtils.ARGUMENT_ELEMENT);
				for (Element argument : argumentsList) {
					String attributeType = argument
							.getAttributeValue(SyntaxUtils.TYPE_ATTRIBUTE);
					if ((attributeType == null)
							|| (attributeType.length() == 0)) {
						attributeType = "void";
					}
					String attributeName = argument
							.getAttributeValue(SyntaxUtils.NAME_ATTRIBUTE);
					displayString += attributeType + " " + attributeName + ", ";
				}
				if (argumentsList.size() > 0) {
					displayString = displayString.substring(0, displayString
							.length() - 2);
				}
			}
			displayString += ")";
			contextInformation = new ContextInformation(completion,
					displayString);
			cursorPosition++;
		}

		String type = element.getAttributeValue(SyntaxUtils.TYPE_ATTRIBUTE);
		if ((type != null) && (type.length() > 0)) {
			displayString += "  " + type;
		}

		String context = element
				.getAttributeValue(SyntaxUtils.CONTEXT_ATTRIBUTE);
		if ((context != null) && (context.length() > 0)) {
			displayString += " (" + context + ")";
		}

		if (completion.startsWith(elementPart)) {
			int replacementLength = elementPart.length();
			int replacementOffset = documentOffset - replacementLength;
			OrderableCompletionProposal completionProposal = new OrderableCompletionProposal(
					completion, replacementOffset, replacementLength,
					cursorPosition, null, displayString, contextInformation,
					null);
			completionProposal.setName(name);
			completionProposal.setFunction(function);
			proposals.add(completionProposal);
		}
	}

	/*
	 * (non-Javadoc) Method declared on IContentAssistProcessor
	 */
	public IContextInformation[] computeContextInformation(ITextViewer viewer,
			int documentOffset) {
		return null;
	}

	/*
	 * (non-Javadoc) Method declared on IContentAssistProcessor
	 */
	public char[] getCompletionProposalAutoActivationCharacters() {
		return new char[] { '.' };
	}

	/*
	 * (non-Javadoc) Method declared on IContentAssistProcessor
	 */
	public char[] getContextInformationAutoActivationCharacters() {
		return new char[] {};
	}

	/*
	 * (non-Javadoc) Method declared on IContentAssistProcessor
	 */
	public IContextInformationValidator getContextInformationValidator() {
		return validator;
	}

	public FramclipseEditor getEditor() {
		return editor;
	}

	/*
	 * (non-Javadoc) Method declared on IContentAssistProcessor
	 */
	public String getErrorMessage() {
		return null;
	}

	/**
	 * Simple content assist tip closer. The tip is valid in a range of 5
	 * characters around its popup location.
	 */
	protected static class Validator implements IContextInformationValidator,
			IContextInformationPresenter {

		protected int installOffset;

		/*
		 * @see IContextInformationValidator#isContextInformationValid(int)
		 */
		public boolean isContextInformationValid(int offset) {
			return Math.abs(installOffset - offset) < 5;
		}

		/*
		 * @see IContextInformationValidator#install(IContextInformation,
		 * ITextViewer, int)
		 */
		public void install(IContextInformation info, ITextViewer viewer,
				int offset) {
			installOffset = offset;
		}

		/*
		 * @see
		 * org.eclipse.jface.text.contentassist.IContextInformationPresenter
		 * #updatePresentation(int, TextPresentation)
		 */
		public boolean updatePresentation(int documentPosition,
				TextPresentation presentation) {
			return false;
		}
	}
}
