// This file is a part of Framsticks SDK.  http://www.framsticks.com/
// Copyright (C) 1999-2021  Maciej Komosinski and Szymon Ulatowski.
// See LICENSE.txt for details.

#include "simil-measure-base.h"
#include <frams/vm/classes/genoobj.h>
#include <frams/model/geometry/meshbuilder.h>
#include <frams/model/geometry/geometryutils.h>

SimilMeasureBase::SimilMeasureBase()
{
	for (int i = 0; i < 2; i++)
	{
		genos[i] = nullptr;
		models[i] = nullptr;
	}
}

double SimilMeasureBase::evaluateDistance(const Geno* G0, const Geno* G1)
{
	genos[0] = G0;
	genos[1] = G1;

	// create models of objects to compare
	models[0] = newModel(genos[0]);
	models[1] = newModel(genos[1]);

	if (models[0] == NULL)
		logPrintf("SimilMeasureBase", "evaluateDistance", LOG_ERROR, "Unable to create a Model from genotype '%s'.", G0->getGenes().c_str());
	if (models[1] == NULL)
		logPrintf("SimilMeasureBase", "evaluateDistance", LOG_ERROR, "Unable to create a Model from genotype '%s'.", G1->getGenes().c_str());
	if (models[0] == NULL || models[1] == NULL)
		return -1;

	double distance = getDistance();
	SAFEDELETE(models[0]);
	SAFEDELETE(models[1]);
	return distance;
}

void SimilMeasureBase::p_evaldistance(ExtValue * args, ExtValue * ret)
{
	Geno * g1 = GenoObj::fromObject(args[1]);
	Geno * g2 = GenoObj::fromObject(args[0]);
	if ((!g1) || (!g2))
		ret->setEmpty();
	else
		ret->setDouble(evaluateDistance(g1, g2));
}

Model* SimilMeasureBase::newModel(const Geno *g)
{
	if (g == NULL)
	{
		logPrintf("SimilarityMeasure", "newModel", LOG_ERROR, "NULL genotype pointer");
		return NULL;
	}
	Model *m = new Model(*g, ModelEnum::SHAPETYPE_UNKNOWN);
	if (!m->isValid())
	{
		logPrintf("SimilarityMeasure", "newModel", LOG_ERROR, "Invalid model for the genotype of '%s'", g->getName().c_str());
		delete m;
		return NULL;
	}
	return m;
}

Model SimilMeasureBase::sampleSurface(Model* M, double density)
{
	Model resultModel;
	resultModel.open();
	GeometryUtils::addAnchorToModel(resultModel);

	MeshBuilder::ModelSurface iterator(density);
	iterator.initialize(M);

	Pt3D point;
	while (iterator.tryGetNext(point))
	{
		GeometryUtils::addPointToModel(point, resultModel);
	}

	resultModel.close();
	return resultModel;
}