// This file is a part of Framsticks SDK.  http://www.framsticks.com/
// Copyright (C) 1999-2020  Maciej Komosinski and Szymon Ulatowski.
// See LICENSE.txt for details.

#include "simil-measure.h"
#include <frams/vm/classes/genoobj.h>
#include <frams/model/geometry/geometryutils.h>
#include <frams/model/geometry/meshbuilder.h>

#define FIELDSTRUCT SimilMeasure

static ParamEntry simil_measure_paramtab[] = {
	{ "Similarity", 1, 2, "SimilMeasure", "Evaluates morphological dissimilarity.", },
	{ "measure_type", 0, 0, "Type of similarity measure", "d 1 3 1", FIELD(measure_type), "", },
	{ "evaluateDistance", 0, PARAM_DONTSAVE | PARAM_USERHIDDEN, "evaluate model dissimilarity", "p f(oGeno,oGeno)", PROCEDURE(p_evaldistance), "Calculates dissimilarity between two models created from Geno objects.", },
	{ 0, },
};

#undef FIELDSTRUCT

SimilMeasure::SimilMeasure() : localpar(simil_measure_paramtab, this)
{
	for (int i = 0; i < 2; i++)
	{
		genos[i] = nullptr;
		models[i] = nullptr;
	}
	localpar.setDefault();
}

double SimilMeasure::evaluateDistance(const Geno* G0, const Geno* G1)
{
	genos[0] = G0;
	genos[1] = G1;

	// create models of objects to compare
	models[0] = newModel(genos[0]);
	models[1] = newModel(genos[1]);

	if (models[0] == NULL || models[1] == NULL)
	{
		logPrintf("SimilarityMeasure", "EvaluateDistance", LOG_ERROR, "Unable to create model from one of the genotypes.");
		return -1;
	}
	
	double distance = getDistance();
	SAFEDELETE(models[0]);
	SAFEDELETE(models[1]);
	return distance;
}

Model* SimilMeasure::newModel(const Geno *g)
{
	if (g == NULL)
	{
		logPrintf("SimilarityMeasure", "newModel", LOG_ERROR, "NULL genotype pointer");
		return NULL;
	}
	Model *m = new Model(*g, ModelEnum::SHAPETYPE_UNKNOWN);
	if (!m->isValid())
	{
		logPrintf("SimilarityMeasure", "newModel", LOG_ERROR, "Invalid model for the genotype of '%s'", g->getName().c_str());
		delete m;
		return NULL;
	}
	return m;
}

void SimilMeasure::p_evaldistance(ExtValue *args, ExtValue *ret)
{
	Geno *g1 = GenoObj::fromObject(args[1]);
	Geno *g2 = GenoObj::fromObject(args[0]);
	if ((!g1) || (!g2))
		ret->setEmpty();
	else
		ret->setDouble(evaluateDistance(g1, g2));
}

Model SimilMeasure::sampleSurface(Model* M, double density)
{
    Model resultModel;
    resultModel.open();
    GeometryUtils::addAnchorToModel(resultModel);

    MeshBuilder::ModelSurface iterator(density);
    iterator.initialize(M);

    Pt3D point;
    while (iterator.tryGetNext(point))
    {
        GeometryUtils::addPointToModel(point, resultModel);
    }

    resultModel.close();
    return resultModel;
}