package com.framsticks.test;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.framsticks.core.ListChange;
import com.framsticks.params.EventListener;
import com.framsticks.params.UniqueListAccess;
import com.framsticks.params.annotations.FramsClassAnnotation;
import com.framsticks.params.annotations.ParamAnnotation;
import com.framsticks.params.types.ProcedureParam;

@FramsClassAnnotation(
	order = {
		"name",
		"history",
		"history_changed",
		"appendHistory",
		"resetHistory",
		"children",
		"createChild",
		"children_changed"
	},
	register = {
		ChangeEvent.class,
		TestChild.class,
		ListChange.class
	}
)
public class TestClass {
	private static final Logger log =
		LogManager.getLogger(TestClass.class);


	protected String name = "test";
	protected String history = "initial|";
	protected final List<EventListener<ChangeEvent>> historyListeners = new LinkedList<>();
	protected final List<EventListener<ListChange>> childrenListeners = new LinkedList<>();

	protected final Map<String, TestChild> children = UniqueListAccess.createMap(TestChild.class, this);
	protected int counter = 0;

	/**
	 *
	 */
	public TestClass() {
	}

	/**
	 * @return the name
	 */
	@ParamAnnotation
	public String getName() {
		return name;
	}

	/**
	 * @param name the name to set
	 */
	@ParamAnnotation
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * @return the history
	 */
	@ParamAnnotation
	public String getHistory() {
		return history;
	}

	/**
	 * @param history the history to set
	 */
	@ParamAnnotation
	public void setHistory(String history) {
		this.history = history;
	}

	/**
	 * @return the children
	 */
	@ParamAnnotation
	public Map<String, TestChild> getChildren() {
		return Collections.unmodifiableMap(children);
	}

	@ParamAnnotation(paramType = ProcedureParam.class)
	public int appendHistory(String line) {
		log.debug("appending '{}'", line);
		history = history + line + "|";
		fireHistoryChange();
		return history.length();
	}

	@ParamAnnotation(paramType = ProcedureParam.class)
	public void resetHistory() {
		log.debug("reseting");
		history = "";
		fireHistoryChange();
	}

	@ParamAnnotation(paramType = ProcedureParam.class)
	public void createChild(String name) {
		TestChild child = new TestChild(this);
		child.name = name;
		children.put(child.getUid(), child);
		fireChildrenChange(child, ListChange.Action.Add);
	}

	protected void fireHistoryChange() {
		for (EventListener<ChangeEvent> l : historyListeners) {
			ChangeEvent event = new ChangeEvent();
			event.history = history;
			l.action(event);
		}
	}

	protected void fireChildrenChange(TestChild child, ListChange.Action action) {
		ListChange change = new ListChange(action, UniqueListAccess.getNumberInMap(children, child), child.getUid());
		for (EventListener<ListChange> l : childrenListeners) {
			l.action(change);
		}
	}

	@ParamAnnotation(id = "history_changed")
	public void addHistoryListener(EventListener<ChangeEvent> listener) {
		historyListeners.add(listener);
	}

	@ParamAnnotation(id = "history_changed")
	public void removeHistoryListener(EventListener<ChangeEvent> listener) {
		historyListeners.remove(listener);
	}

	@ParamAnnotation(id = "children_changed")
	public void addChildrenListener(EventListener<ListChange> listener) {
		childrenListeners.add(listener);
	}

	@ParamAnnotation(id = "children_changed")
	public void removeChildrenListener(EventListener<ListChange> listener) {
		childrenListeners.remove(listener);
	}

	@Override
	public String toString() {
		return "test class " + history;
	}

}
