package com.framsticks.params;


import static com.framsticks.util.lang.Containers.filterInstanceof;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.framsticks.params.types.ObjectParam;
import com.framsticks.util.FramsticksException;
import static com.framsticks.params.SetStateFlags.*;

/**
 * The Class SimpleAbstractAccess implements all the methods of Access
 * which actions can be implemented with usage of {@link Access} methods
 * or concern schema, which is stored in {@link #framsClass}
 *
 * Based on c++ class SimpleAbstractParam located in: cpp/gdk/param.*
 *
 * @author Jarek Szymczak <name.surname@gmail.com>, Mateusz Jarus (please
 *         replace name and surname with my personal data)
 *
 * @author Piotr Sniegowski
 */
public abstract class SimpleAbstractAccess implements Access {

	private final static Logger log = LogManager.getLogger(SimpleAbstractAccess.class.getName());

	protected final FramsClass framsClass;

	/**
	 * @param framsClass
	 */
	public SimpleAbstractAccess(FramsClass framsClass) {
		this.framsClass = framsClass;
	}

	@Override
	public final FramsClass getFramsClass() {
		return framsClass;
	}

	/**
	 * Simple String key, value class.
	 */
	public static class Entry {

		public final String key;
		public final String value;

		public Entry(String key, String value) {
			this.key = key;
			this.value = value;
		}

		@Override
		public String toString() {
			return key + " = " + value;
		}
	}


	@Override
	public String getId() {
		return framsClass.getId();
	}

	@Override
	public int getParamCount() {
		return framsClass.getParamCount();
	}

	@Override
	public Param getParam(int i) {
		return framsClass.getParam(i);
	}

	@Override
	public Param getParam(String id) {
		return framsClass.getParam(id);
	}

	// @Override
	// public Param getGroupMember(int gi, int n) {
	// 	return framsClass.getGroupMember(gi, n);
	// }

	@Override
	public <T> T get(int i, Class<T> type) {
		return get(framsClass.getParamEntry(i, ValueParam.class), type);
	}

	@Override
	public <T> T get(String id, Class<T> type) {
		return get(framsClass.getParamEntry(id, ValueParam.class), type);
	}

	@Override
	public <T> int set(int i, T value) {
		return set(framsClass.getParamEntry(i, ValueParam.class), value);
	}

	@Override
	public <T> int set(String id, T value) {
		return set(framsClass.getParamEntry(id, ValueParam.class), value);
	}

	@Override
	public <T> int set(ValueParam param, T value) {

		//String id = param.getEffectiveId();
		try {
			Object oldValue = get(param, param.getStorageType());
			ReassignResult<?> result = param.reassign(value, oldValue);
			Object casted = result.getValue();
			if (casted != null && !casted.equals(oldValue)) {
				internalSet(param, casted);
			}
			return result.getFlags();
		} catch (CastFailure e) {
			throw new FramsticksException()
				.msg("casting failure while set")
				.arg("param", param)
				.arg("value", value)
				.arg("value's type", (value == null ? "<null>" : value.getClass().getCanonicalName()))
				.arg("in", this).cause(e);
		}
	}

	@Override
	public void setDefault(boolean numericOnly) {
		for (int i = 0; i < framsClass.getParamCount(); i++) {
			setDefault(i, numericOnly);
		}
	}

	@Override
	public void setDefault(int i, boolean numericOnly) {
		ValueParam entry = framsClass.getParamEntry(i, ValueParam.class);
		if ((entry != null)	&& (!numericOnly || entry.isNumeric())) {
			set(i, entry.getDef(entry.getStorageType()));
		}
	}

	@Override
	public void setMin() {
		for (int i = 0; i < framsClass.getParamCount(); i++) {
			setMin(i);
		}
	}

	@Override
	public void setMin(int i) {
		PrimitiveParam<?> entry = framsClass.getParamEntry(i, PrimitiveParam.class);
		Object min = entry.getMin(entry.getStorageType());
		if (min != null) {
			set(i, min);
		}
	}

	@Override
	public void setMax() {
		for (int i = 0; i < framsClass.getParamCount(); i++) {
			setMax(i);
		}
	}

	@Override
	public void setMax(int i) {
		PrimitiveParam<?> entry = framsClass.getParamEntry(i, PrimitiveParam.class);
		Object max = entry.getMax(entry.getStorageType());
		if (max != null) {
			set(i, max);
		}
	}

	@Override
	public void save(SinkInterface sink) {
		assert framsClass != null;
		sink.print(framsClass.getId()).print(":").breakLine();
		for (PrimitiveParam<?> p : filterInstanceof(framsClass.getParamEntries(), PrimitiveParam.class)) {
			Object value = get(p, Object.class);
			if ((value == null) || value.equals(p.getDef(Object.class))) {
				continue;
			}
			sink.print(p.getId()).print(":");
			p.save(sink, value);
			sink.breakLine();
		}
		sink.breakLine();
	}

	private static Entry readEntry(SourceInterface source) {

		String line;
		String key = null;
		StringBuilder value = null;
		while ((line = source.readLine()) != null) {
			if (key == null) {
				int colonIndex = line.indexOf(':');
				if (colonIndex == -1) {
					return null;
				}
				key = line.substring(0, colonIndex);
				String inlineValue = line.substring(colonIndex + 1);


				if (!inlineValue.startsWith("~")) {
					return new Entry(key, inlineValue);
				}
				value = new StringBuilder();
				value.append(inlineValue.substring(1));
				continue;
			}
			if (value.length() != 0) {
				value.append(System.getProperty("line.separator"));
			}
			if (line.endsWith("~") && !line.endsWith("\\~")) {
				value.append(line.substring(0, line.length() - 1));
				return new Entry(key, value.toString().replaceAll("\\\\~", "~"));
			}
			value.append(line);
		}
		return null;
	}

	@Override
	public void load(SourceInterface source) {
		//TODO not clearing values, because get from manager gives only fields, not children
		//this.clearValues();

		Entry entry;
		while ((entry = readEntry(source)) != null) {
			Param param = getParam(entry.key);
			if (param == null) {
				throw new FramsticksException().msg("param not found in access").arg("name", entry.key).arg("access", this);
			}
			if (!(param instanceof ValueParam)) {
				throw new FramsticksException().msg("param is not a value param").arg("param", param).arg("access", this);
			}
			if ((param.getFlags() & ParamFlags.DONTLOAD) == 0) {
				int retFlags = this.set((ValueParam) param, entry.value);
				if ((retFlags & (PSET_HITMIN | PSET_HITMAX)) != 0) {
					String which = ((retFlags & PSET_HITMIN) != 0) ? "small" : "big";
					log.warn("value of key '{}' was too {}, adjusted", entry.key, which);
				}
			}
		}
	}

	protected abstract <T> void internalSet(ValueParam param, T value);

	@Override
	public Iterable<Param> getParams() {
		return framsClass.getParamEntries();
	}

	@Override
	public int getCompositeParamCount() {
		return framsClass.getCompositeParamCount();
	}

	@Override
	public CompositeParam getCompositeParam(int number) {
		return framsClass.getCompositeParam(number);
	}

	/*
	protected <T extends Comparable<T>> int setAndCut(Param param, Object value, Class<T> type) {
		int flags = 0;
		T val = type.cast(value);
		T min = param.getMin(type);
		T max = param.getMax(type);
		if (min != null && val.compareTo(min) < 0) {
			val = min;
			flags |= Flags.PSET_HITMIN;
		}
		if (max != null && val.compareTo(max) > 0) {
			val = max;
			flags |= Flags.PSET_HITMAX;
		}
		internalSet(param, val);
		return flags;
	}*/


	@Override
	public ParamBuilder buildParam(ParamBuilder builder) {
		return builder.name(getId()).type(ObjectParam.class).containedTypeName(getId());
	}

}
