package com.framsticks.params;

import com.framsticks.params.annotations.FramsClassAnnotation;
import com.framsticks.params.annotations.ParamAnnotation;
import com.framsticks.util.FramsticksException;
import com.framsticks.util.lang.Containers;
import com.framsticks.util.lang.Strings;
// import com.framsticks.util.FramsticksException;

import java.util.*;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.Immutable;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

/**
 * The class FramsClass represents the class / schema of connected parameters
 * (such as parameters within the class). It differs from C++ version by storing
 * information about the class that parameters belong to.
 *
 * Based loosely on c++ class Param located in cpp/gdk/param.*
 *
 * @author Jarek Szymczak <name.surname@gmail.com>, Mateusz Jarus (please
 *         replace name and surname with my personal data)
 *
 * @author Piotr Sniegowski
 */
@Immutable
@FramsClassAnnotation(id = "class", name = "class")
public class FramsClass {

	private final static Logger log = LogManager.getLogger(FramsClass.class);

	protected final String id;

	protected final String name;

	protected final String description;

	protected final List<Group> groups;

	/** The param list (for accessing parameters by offset in O(1) time. */
	protected final List<Param> paramList;

	protected final List<CompositeParam> compositeParamList = new ArrayList<>();

	/**
	 * The param entry map <parameterId, param> (for fast accessing of parameters
	 * by their name)
	 */
	protected Map<String, Param> paramEntryMap = new HashMap<>();


	@ParamAnnotation(id = "props", name = "props")
	public List<Param> getParamEntries() {
		return Collections.unmodifiableList(paramList);
	}

	public FramsClass(FramsClassBuilder builder) {

		this.id = builder.getId();
		this.name = Strings.toStringEmptyProof(builder.getName(), this.id);
		this.description = builder.getDescription();
		this.groups = Containers.build(builder.groupBuilders);
		this.paramList = builder.params;

		for (Param param : paramList) {
			paramEntryMap.put(param.getId(), param);
			if (param instanceof CompositeParam) {
				compositeParamList.add((CompositeParam) param);
			}
		}

		log.trace("created framsclass {}", this);

	}

	@ParamAnnotation(id = "desc")
	public String getDescription() {
		return description;
	}

	public int getGroupCount() {
		return groups.size();
	}

	public Group getGroup(int groupNumber) {
		return Containers.getFromList(groups, groupNumber, "group", this);
	}

	// /**
	//  * Gets the group member.
	//  *
	//  * @param gi
	//  *            the offset of group
	//  * @param pi
	//  *            the offset of member within a group
	//  * @return the pi-th member of group gi
	//  */
	// public Param getGroupMember(int gi, int pi) {
	// 	if (gi < 0 || pi < 0 || gi >= groups.size()) {
	// 		return null;
	// 	}
	// 	Group group = groups.get(gi);
	// 	return (group != null ? group.getProperty(pi) : null);
	// }

	// /**
	//  * Gets the group name.
	//  *
	//  * @param gi
	//  *            the offset of group
	//  * @return the group name
	//  */
	// public String getGroupName(int gi) {
	// 	if (gi < 0 || gi >= groups.size())
	// 		return null;
	// 	return groups.get(gi).name;
	// }

	@ParamAnnotation
	public String getId() {
		return id;
	}

	@ParamAnnotation
	public String getName() {
		return name;
	}

	public String getNiceName() {
		return name != null ? name : id;
	}

	public @Nonnull <T extends Param> T castedParam(@Nonnull final Param param, @Nonnull final Class<T> type, Object name) {
		if (param == null) {
			// return null;
			throw new FramsticksException().msg("param is missing").arg("name", name).arg("in", this);
		}
		if (!type.isInstance(param)) {
			// return null;
			throw new FramsticksException().msg("wrong type of param").arg("actual", param.getClass()).arg("requested", type).arg("in", this);
		}
		return type.cast(param);
	}

	/**
	 * Gets the param entry.
	 *
	 * @param i
	 *            the offset of parameter
	 * @return the param entry
	 */
	public @Nonnull <T extends Param> T getParamEntry(final int i, @Nonnull final Class<T> type) {
		return castedParam(getParam(i), type, i);
	}

	/**
	 * Gets the param entry.
	 *
	 * @param id
	 *            the getId of parameter
	 * @return the param entry
	 */
	public @Nonnull <T extends Param> T getParamEntry(@Nonnull final String id, @Nonnull final Class<T> type) {
		return castedParam(getParam(id), type, id);
	}

	public Param getParam(int i) {
		if (i < 0 || i >= paramList.size()) {
			return null;
		}
		return paramList.get(i);
	}

	public Param getParam(String id) {
		if (!paramEntryMap.containsKey(id)) {
			return null;
		}
		return paramEntryMap.get(id);
	}

	public int getParamCount() {
		return paramList.size();
	}

	public final int getCompositeParamCount() {
		return compositeParamList.size();
	}

	public final CompositeParam getCompositeParam(int i) {
		return compositeParamList.get(i);
	}

	@Override
	public String toString() {
		return id + "(" + name + ")";
	}

	public static FramsClassBuilder build() {
		return new FramsClassBuilder();
	}

}
