package com.framsticks.gui.table;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JTable;
import javax.swing.UIManager;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.framsticks.params.Access;
import com.framsticks.params.ListAccess;
import com.framsticks.params.Param;
import com.framsticks.params.PrimitiveParam;
import com.framsticks.params.types.ProcedureParam;
import com.framsticks.util.lang.Casting;

public class TableModel implements javax.swing.table.TableModel {

	private static final Logger log = LogManager.getLogger(TableModel.class);

	protected List<TableModelListener> listeners = new LinkedList<>();
	protected ListAccess access;
	protected Access elementAccess;
	protected final List<Column> columns = new ArrayList<>();
	protected final ListPanel listPanel;
	protected JTable table;

	/**
	 * @param listPanel
	 */
	public TableModel(ListPanel listPanel) {
		this.listPanel = listPanel;
	}

	/**
	 * @return the listPanel
	 */
	public ListPanel getListPanel() {
		return listPanel;
	}

	public JTable getTable() {
		return listPanel.getTable();
	}

	protected void refreshAll() {
		for (TableModelListener l : listeners) {
			l.tableChanged(new TableModelEvent(this));
		}
	}

	/**
	 * @param table the table to set
	 */
	public void setupTable() {
		getTable().setDefaultRenderer(ProcedureParam.class, new ProcedureRenderer());
		getTable().setDefaultEditor(ProcedureParam.class, new ProcedureEditor(new JCheckBox()));
	}

	public void attachSource(ListAccess access) {
		this.access = Casting.assertCast(ListAccess.class, access.cloneAccess().select(access.getSelected()));

		this.elementAccess = this.access.getElementAccess().cloneAccess();
		log.debug("attached {}", access);
		refreshAll();
	}

	@Override
	public void addTableModelListener(TableModelListener listener) {
		listeners.add(listener);
	}

	@Override
	public Class<?> getColumnClass(int columnIndex) {
		// log.debug("returning column type {}", columnParams.get(columnIndex).getStorageType());
		return columns.get(columnIndex).getColumnClass();
	}

	@Override
	public int getColumnCount() {
		// log.debug("returning column count {}", columnParams.size());
		return columns.size();
	}

	@Override
	public String getColumnName(int columnIndex) {
		return columns.get(columnIndex).getParam().getName();
	}

	@Override
	public int getRowCount() {
		// log.debug("returning row count {}", access.getParamCount());
		return access == null ? 0 : access.getParamCount();
	}

	@Override
	public Object getValueAt(int rowIndex, int columnIndex) {
		assert (rowIndex < access.getParamCount() && columnIndex < columns.size());

		return columns.get(columnIndex).getValueAt(rowIndex);
	}

	@Override
	public boolean isCellEditable(int rowIndex, int columnIndex) {
		return columns.get(columnIndex).isEditable();
	}

	@Override
	public void removeTableModelListener(TableModelListener listener) {
		listeners.remove(listener);
	}


	@Override
	public void setValueAt(Object value, int rowIndex, int columnIndex) {
		assert (rowIndex < access.getParamCount() && columnIndex < columns.size());
		// Object object = getObjectAt(rowIndex, columnIndex);
		// if (object == null) {
		//	return;
		// }
		columns.get(columnIndex).setValueAt(rowIndex, value);
	}

	public void addColumn(Column column) {
		log.debug("added {}", column);
		columns.add(column);
	}

	/**
	 * @return the access
	 */
	public ListAccess getAccess() {
		return access;
	}

	/**
	 * @return the elementAccess
	 */
	public Access getElementAccess() {
		return elementAccess;
	}

	/**
	 * @return the columnParams
	 */
	public List<Column> getColumns() {
		return Collections.unmodifiableList(columns);
	}


	public boolean addColumnIfSupported(Param param) {
		if (param instanceof PrimitiveParam) {
			addColumn(new PrimitiveColumn((PrimitiveParam<?>) param, this));
			return true;
		}
		if (param instanceof ProcedureParam) {
			if (((ProcedureParam) param).getArgumentsType().size() == 0) {
				addColumn(new ProcedureColumn((ProcedureParam) param, this));
				return true;
			}
			return false;
		}
		return false;
	}

	public static void transferCellAppeariance(JTable table, JComponent component, boolean isSelected) {
		if (isSelected) {
			component.setForeground(table.getSelectionForeground());
			component.setBackground(table.getSelectionBackground());
		} else {
			component.setForeground(table.getForeground());
			component.setBackground(UIManager.getColor("Button.background"));
		}

	}

}
