package com.framsticks.gui;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.framsticks.core.Tree;
import com.framsticks.core.Node;
import com.framsticks.core.Path;
import com.framsticks.core.TreeOperations;
import com.framsticks.gui.controls.ValueControl;
import com.framsticks.params.CompositeParam;
import com.framsticks.params.FramsClass;

import java.util.*;



import com.framsticks.util.dispatching.FutureHandler;

/**
 * @author Piotr Sniegowski
 */
public class TreeAtFrame {

	private static final Logger log = LogManager.getLogger(TreeAtFrame.class);

	protected final Frame frame;
	protected final Tree tree;
	protected final Map<String, TreePanel> knownPanels = new HashMap<>();
	protected Node rootNode;

	public TreeAtFrame(Tree tree, Frame frame) {
		this.frame = frame;
		this.tree = tree;
	}

	public Frame getFrame() {
		return frame;
	}

	/**
	 * @return the tree
	 */
	public Tree getTree() {
		return tree;
	}

	public final String getName() {
		return tree.getName();
	}

	public TreePanel preparePanel(final CompositeParam param) {
		assert frame.isActive();

		TreePanel panel = knownPanels.get(param.getFramsTypeName());
		if (panel != null) {
			return panel;
		}

		final FramsClass framsClass = tree.getInfoFromCache(param.getContainedTypeName());
		final List<TreePanel> panels = new ArrayList<TreePanel>();

		final TreePanel.Parameters parameters = new TreePanel.Parameters(this, param, framsClass);
		for (PanelProvider pp : frame.browser.panelProviders) {
			TreePanel p = pp.providePanel(parameters);
			if (p != null) {
				panels.add(p);
			}
		}

		if (panels.isEmpty()) {
			panel = new EmptyTreePanel(parameters);
		} else 	if (panels.size() == 1) {
			panel = panels.get(0);
		} else {
			panel = new MultiPanel(parameters, panels);
		}

		knownPanels.put(param.getFramsTypeName(), panel);

		log.debug("prepared panel for {}", panel);
		return panel;
	}


	public boolean hasLocalChanges(Object object) {
		NodeAtFrame nodeAtFrame = tree.getSideNote(object, this, NodeAtFrame.class);
		if (nodeAtFrame == null) {
			return false;
		}
		return !nodeAtFrame.localChanges.isEmpty();
	}

	public NodeAtFrame assureLocalInfo(Object object) {
		assert frame.isActive();
		NodeAtFrame nodeAtFrame = tree.getSideNote(object, this, NodeAtFrame.class);

		if (nodeAtFrame == null) {
			nodeAtFrame = new NodeAtFrame();
			// log.debug();
			tree.putSideNote(object, this, nodeAtFrame);
		}
		return nodeAtFrame;
	}

	public NodeAtFrame getLocalInfo(Object object) {
		return tree.getSideNote(object, this, NodeAtFrame.class);
	}

	public boolean changeValue(Object object, ValueControl component, Object newValue) {
		log.debug("changing value of {} to '{}'", component, newValue);

		assureLocalInfo(object).localChanges.put(component, newValue);

		return true;
	}

	public void pushLocalChanges(Path path) {
		assert frame.isActive();
		path.assureResolved();

		NodeAtFrame nodeAtFrame = getLocalInfo(path.getTopObject());
		if (nodeAtFrame == null) {
			return;
		}
		for (Map.Entry<ValueControl, Object> e : nodeAtFrame.localChanges.entrySet()) {
			TreeOperations.set(path, e.getKey().getParam(), e.getValue(), new FutureHandler<Integer>(frame) {
				@Override
				protected void result(Integer flag) {
				}
			});
		}
	}

}
