package com.framsticks.core;

import java.util.ArrayList;

import org.testng.annotations.BeforeClass;
// import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import com.framsticks.model.Joint;
import com.framsticks.model.Model;
import com.framsticks.model.ModelBuilder;
import com.framsticks.model.f0.Schema;
import com.framsticks.model.f0.SchemaBuilder;
import com.framsticks.test.TestConfiguration;
import com.framsticks.util.dispatching.Dispatching;
import com.framsticks.util.dispatching.Monitor;
import com.framsticks.util.dispatching.RunAt;
import com.framsticks.params.Access;
import com.framsticks.params.ReflectionAccess;
import com.framsticks.params.types.FloatParam;

import static com.framsticks.core.TreeOperations.*;


import static org.fest.assertions.Assertions.*;

@Test
public class LocalTreeTest extends TestConfiguration {

	private Model model;
	private Schema schema;
	private LocalTree tree;
	private Monitor monitor;

	@BeforeClass
	public void setUp() {
		schema = new SchemaBuilder().stream(Schema.getDefaultDefinitionAsStream()).finish();
		assertThat(schema.getFramsClasses().size()).isEqualTo(5);
		assertThat(schema.getNeuroClasses().size()).isEqualTo(21);
	}

	@Test
	public void loadModel() {
		model = new ModelBuilder().schema(schema).stream(LocalTreeTest.class.getResourceAsStream("/parsers/f0_example.txt")).finish();

		assertThat(model.getJoints().size()).isEqualTo(2);
	}


	@Test(dependsOnMethods = "loadModel", timeOut = 2000)
	public void startTree() {
		tree = new LocalTree();
		tree.takeAllFrom(schema.getRegistry());

		tree.setRootObject(model);

		monitor = new Monitor(tree);
		monitor.use();

		assertThat(Dispatching.get(tree, new Dispatching.QueryHandler<String>(failOnException) {
				@Override
				public String get() {
					return tree.getRootObject(Model.class).getParts().get(2).info;
				}
			})).isEqualTo("bla");

		tree.dispatch(new RunAt<Tree>(failOnException) {
			@Override
			protected void runAt() {
				assertThat(tree.getRootObject(Model.class).getNeuroDefinitions().get(2).details).isEqualTo("G");

				Path path = Path.to(tree, "/");
				assertThat(path.isResolved());
				Access access = TreeOperations.bindAccess(path);
				assertThat(access.get("se", Double.class)).isEqualTo(1.0);

				assertThat(bindAccess(tree, "/parts/2").getFramsClass().getParamEntry("m", FloatParam.class).getDef(Double.class)).isEqualTo(1.0);
				assertThat(bindAccess(tree, "/parts/2").getFramsClass().getParamEntry("m", FloatParam.class).getMax(Double.class)).isEqualTo(999.0);
				assertThat(bindAccess(tree, "/parts/2")).isInstanceOf(ReflectionAccess.class);
				assertThat(Path.to(tree, "/neurodefs").getTopObject()).isInstanceOf(ArrayList.class);
				assertThat(Path.to(tree, "/joints/1").getTopObject()).isInstanceOf(Joint.class);
				assertThat(bindAccess(tree, "/parts/2").get("i", String.class)).isEqualTo("bla");
			}
		});

	}

	@Test(dependsOnMethods = "startTree", timeOut = 2000)
	public void stopTree() {
		monitor.useFor(1.0).drop().join();
	}
}
