package com.framsticks.util.dispatching;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import com.framsticks.util.dispatching.Dispatching;
import java.lang.Thread;

public class Monitor implements JoinableParent {
	private static final Logger log =
		LogManager.getLogger(Monitor.class);

	protected final Joinable joinable;
	protected final Thread shutdownHook;

	/**
	 * @param joinable
	 */
	public Monitor(Joinable joinable) {
		this.joinable = joinable;

		shutdownHook = new Thread(new Runnable() {
			@Override
			public void run() {
				log.debug("running shutdown hook");
				Monitor.this.drop().join();
			}
		});
	}

	public Monitor use() {
		Runtime.getRuntime().addShutdownHook(shutdownHook);

		log.debug("{} is using", this);
		Dispatching.use(joinable, this);
		return this;
	}

	public Monitor useFor(double seconds) {
		Dispatching.sleep(seconds);
		return this;
	}

	public Monitor waitFor() {
		log.debug("{} is waiting", this);
		synchronized (this) {
			while (joinable.getState().ordinal() < JoinableState.FINISHING.ordinal()) {
				Dispatching.wait(this, 100);
			}
		}
		log.debug("{} ended waiting", this);
		return this;
	}


	public Monitor drop() {
		log.debug("{} is droping", this);
		Dispatching.drop(joinable, this);
		return this;
	}

	public Monitor join() {
		log.debug("{} is joining", this);
		Dispatching.joinAbsolutely(joinable);
		log.debug("{} is joined", this);

		try {
			Runtime.getRuntime().removeShutdownHook(shutdownHook);
		} catch (IllegalStateException e) {
			/** In case IllegalStateException is caught, it means that JVM is in finalization stage */
		}

		return this;
	}

	@Override
	// @SuppressWarnings("NN_NAKED_NOTIFY")
	public void childChangedState(Joinable joinable, JoinableState state) {
		synchronized (this) {
			this.notify();
		}
		log.debug("{} received notification about transition to {}", this, state);
	}

	@Override
	public String toString() {
		return "monitor for " + joinable;
	}

	@Override
	public Monitor getMonitor() {
		return this;
	}

	protected ExceptionHandler taskExceptionHandler;

	/**
	 * @return the taskExceptionHandler
	 */
	public ExceptionHandler getTaskExceptionHandler() {
		return taskExceptionHandler;
	}

	/**
	 * @param taskExceptionHandler the taskExceptionHandler to set
	 */
	public void setTaskExceptionHandler(ExceptionHandler taskExceptionHandler) {
		this.taskExceptionHandler = taskExceptionHandler;
	}

}
