package com.framsticks.core;


import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nonnull;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.framsticks.communication.File;
import com.framsticks.params.Access;
import com.framsticks.params.CompositeParam;
import com.framsticks.params.EventListener;
import com.framsticks.params.FramsClass;
import com.framsticks.params.ListAccess;
import com.framsticks.params.Param;
import com.framsticks.params.ParamBuilder;
import com.framsticks.params.PrimitiveParam;
import com.framsticks.params.PropertiesAccess;
import com.framsticks.params.UniqueListAccess;
import com.framsticks.params.Util;
import com.framsticks.params.types.EventParam;
import com.framsticks.params.types.ObjectParam;
import com.framsticks.params.types.ProcedureParam;
import com.framsticks.parsers.Loaders;
import com.framsticks.parsers.MultiParamLoader;
import com.framsticks.util.FramsticksException;
import com.framsticks.util.dispatching.Dispatching;
import com.framsticks.util.dispatching.Future;
import com.framsticks.util.dispatching.FutureHandler;
import com.framsticks.util.dispatching.RunAt;

import static com.framsticks.util.dispatching.Dispatching.*;

public final class TreeOperations {

	private static final Logger log = LogManager.getLogger(TreeOperations.class);

	private TreeOperations() {
	}

	public static final Object FETCHED_MARK = new Object();

	public static @Nonnull FramsClass processFetchedInfo(Tree tree, File file) {
		assert tree.isActive();
		FramsClass framsClass = Loaders.loadFramsClass(file.getContent());
		log.debug("process fetched info for {}: {}", tree, framsClass);
		tree.putInfoIntoCache(framsClass);
		return framsClass;
	}

	public static Path create(Path path) {
		assert !path.isResolved();

		Access access = path.getTree().prepareAccess(path.getTop().getParam());
		Object child = createAccessee(path.getTree(), access);
		assert child != null;
		if (path.size() == 1) {
			path.getTree().assignRootObject(child);
		} else {
			Access parentAccess = bindAccess(path.getUnder());

			/** this special case is not very good - maybe hide it in createAccessee? */
			if (parentAccess instanceof UniqueListAccess) {
				access.select(child);
				access.set(((UniqueListAccess) parentAccess).getUidName(), path.getTop().getParam().getId());
			}

			parentAccess.set(path.getTop().getParam(), child);
		}
		path = path.appendResolution(child);
		return path;
	}

	public static void processFetchedValues(final Path path, final List<File> files, final Access access, final Future<Path> future) {
		assert files.size() == 1;
		assert path.isTheSame(files.get(0).getPath());


		try {
			log.debug("process fetched values: {}", path);
			final Access parsingAccess = new PropertiesAccess(access.getFramsClass());
			final List<Object> results = MultiParamLoader.loadAll(files.get(0).getContent(), parsingAccess);

			Dispatching.dispatchIfNotActive(path.getTree(), new RunAt<Tree>(future) {
				@Override
				protected void runAt() {

					Path result = path.tryResolveIfNeeded();

					if (!result.isResolved()) {
						result = create(result);
					}

					if (path.getTop().getParam() instanceof ObjectParam) {
						assert results.size() == 1;
						Util.takeAllNonNullValues(bindAccess(result), parsingAccess.select(results.get(0)));
						mark(result.getTree(), result.getTopObject(), FETCHED_MARK, true);
						future.pass(result);
						return;
					}


					final ListAccess listAccess = (ListAccess) access;

					listAccess.select(result.getTopObject());
					Set<String> oldValuesIds = new HashSet<>();
					for (Param p : listAccess.getParams()) {
						oldValuesIds.add(p.getId());
					}

					Access targetAccess = listAccess.getElementAccess();//.cloneAccess();

					int number = 0;
					for (Object r : results) {

						parsingAccess.select(r);
						String id;
						if (listAccess instanceof UniqueListAccess) {
							id = parsingAccess.get(((UniqueListAccess) listAccess).getUidName(), String.class);
						} else {
							id = Integer.toString(number);
						}
						++number;

						Object childTo = listAccess.get(id, Object.class);
						boolean newOne;
						if (childTo == null) {
							childTo = createAccessee(result.getTree(), targetAccess);
							newOne = true;
						} else {
							assert oldValuesIds.contains(id);
							newOne = false;
						}
						oldValuesIds.remove(id);

						targetAccess.select(childTo);
						Util.takeAllNonNullValues(targetAccess, parsingAccess);
						if (newOne) {
							listAccess.set(id, childTo);
						}
						mark(result.getTree(), childTo, FETCHED_MARK, true);

					}
					mark(result.getTree(), result.getTopObject(), FETCHED_MARK, true);

					/** It looks tricky for ArrayListAccess but should also work.
					 *
					 * They should be sorted.
					 */
					for (String id : oldValuesIds) {
						listAccess.set(id, null);
					}
					future.pass(result);
				}
			});

		} catch (FramsticksException e) {
			throw new FramsticksException().msg("exception occurred while loading").cause(e);
		}
	}

	public static FramsClass getInfo(Path path) {
		Tree tree = path.getTree();
		assert tree.isActive();
		log.debug("get info for: {}", path);
		final String name = path.getTop().getParam().getContainedTypeName();
		return tree.getInfoFromCache(name);
	}

	public static void findInfo(final Path path, final Future<FramsClass> future) {
		log.debug("find info for: {}", path);
		try {
			Tree tree = path.getTree();
			assert tree.isActive();
			final FramsClass framsClass = getInfo(path);
			if (framsClass != null) {
				future.pass(framsClass);
				return;
			}
			tree.info(path, future);
		} catch (FramsticksException e) {
			future.handle(e);
		}
	}


	public static @Nonnull Access bindAccessFromSideNote(Tree tree, Object object) {
		CompositeParam param = tree.getSideNote(object, CompositeParam.class, CompositeParam.class);
		if (param == null) {
			throw new FramsticksException().msg("failed to bind access from side node").arg("tree", tree).arg("object", object).arg("type", object.getClass());
		}
		return tree.prepareAccess(param).select(object);
	}

	public static @Nonnull Access bindAccess(Tree tree, String path) {
		log.debug("bind access for textual: {} in {}", path, tree);
		return bindAccess(Path.to(tree, path));
	}

	public static @Nonnull Access bindAccess(Node node) {
		Tree tree = node.getTree();
		assert tree.isActive();
		assert node.getObject() != null;

		try {
			Access access = tree.prepareAccess(node.getParam());
			tree.putSideNote(node.getObject(), CompositeParam.class, node.getParam());

			return access.select(node.getObject());
		} catch (FramsticksException e) {
			throw new FramsticksException().msg("failed to prepare access for param").arg("param", node.getParam()).cause(e);
			// log.error("failed to bind access for {}: ", node.getParam(), e);
		}
	}

	public static @Nonnull Access bindAccess(Path path) {
		assert path.getTree().isActive();
		path.assureResolved();
		log.debug("bind access for: {}", path);
		return bindAccess(path.getTop());
	}

	public static void set(final Path path, final PrimitiveParam<?> param, final Object value, final Future<Integer> future) {
		final Tree tree = path.getTree();

		dispatchIfNotActive(tree, new RunAt<Tree>(future) {
			@Override
			protected void runAt() {
				tree.set(path, param, value, future);
			}
		});
	}

	public static void call(final Path path, final String procedureName, final Object[] arguments, final Future<Object> future) {
		final Tree tree = path.getTree();

		dispatchIfNotActive(tree, new RunAt<Tree>(future) {
			@Override
			protected void runAt() {
				path.assureResolved();
				tree.call(path, tree.getRegistry().getFramsClass(path.getTop().getParam()).getParamEntry(procedureName, ProcedureParam.class), arguments, future);
			}
		});
	}

	public static void call(final Path path, final ProcedureParam param, final Object[] arguments, final Future<Object> future) {
		final Tree tree = path.getTree();

		dispatchIfNotActive(tree, new RunAt<Tree>(future) {
			@Override
			protected void runAt() {
				tree.call(path, param, arguments, future);
			}
		});
	}

	public static <A> void addListener(final Path path, final EventParam param, final EventListener<A> listener, final Class<A> argument, final Future<Void> future) {
		final Tree tree = path.getTree();

		dispatchIfNotActive(tree, new RunAt<Tree>(future) {
			@Override
			protected void runAt() {
				tree.addListener(path, param, listener, argument, future);
			}
		});
	}

	public static void removeListener(final Path path, final EventParam param, final EventListener<?> listener, final Future<Void> future) {
		final Tree tree = path.getTree();

		dispatchIfNotActive(tree, new RunAt<Tree>(future) {
			@Override
			protected void runAt() {
				tree.removeListener(path, param, listener, future);
			}
		});
	}


	/**
	 *
	 * If StackOverflow occurs in that loop in LocalTree it is probably caused
	 * the by the fact, that get operation may find the object, but Path resolution
	 * cannot.
	 * */
	public static void tryGet(final Tree tree, final String targetPath, final Future<Path> future) {
		log.debug("resolve textual: {} for {}", targetPath, tree);
		dispatchIfNotActive(tree, new RunAt<Tree>(future) {

			@Override
			protected void runAt() {
				final Path path = Path.tryTo(tree, targetPath).tryResolveIfNeeded();
				log.debug("found: {}", path);
				if (path.isResolved()) {
					future.pass(path);
					return;
				}

				tree.get(path, new FutureHandler<Path>(future) {
					@Override
					protected void result(Path result) {
						// if (result.isResolved(targetPath)) {
						//	future.pass(result);
						//	return;
						// }
						log.debug("retrying resolve textual: {} for {} with {}", targetPath, tree, result);
						tryGet(tree, targetPath, future);
					}
				});
			}
		});
	}

	public static FramsClass getInfoFromCache(Path path) {
		assert path.getTree().isActive();
		return path.getTree().getInfoFromCache(path.getTop().getParam().getContainedTypeName());
	}

	public static Object createAccessee(Tree tree, CompositeParam param) {
		Object object = tree.prepareAccess(param).createAccessee();
		tree.putSideNote(object, CompositeParam.class, param);
		return object;
	}

	public static Object createAccessee(Tree tree, Access access) {
		Object object = access.createAccessee();
		tree.putSideNote(object, CompositeParam.class, access.buildParam(new ParamBuilder()).finish(CompositeParam.class));
		return object;
	}

	public static boolean isMarked(Tree tree, Object object, Object mark, boolean defValue) {
		assert tree.isActive();
		Boolean v = tree.getSideNote(object, mark, Boolean.class);
		return (v != null ? v : defValue);
	}

	public static void mark(Tree tree, Object object, Object mark, boolean value) {
		assert tree.isActive();
		tree.putSideNote(object, mark, value);
	}

	public static FramsClass getFramsClass(Path path) {
		return path.getTree().getRegistry().getFramsClass(path.getTop().getParam());
	}

}
